(ns clj-vw.rox
  (:import [java.io IOException]
           [java.net InetAddress InetSocketAddress Socket]
           [java.nio ByteBuffer]
           [java.nio.channels SelectionKey Selector SocketChannel]
           [java.nio.channels.spi SelectorProvider]))

(def REGISTER 1)

(defn change-request [socket-channel type ops]
  ;; public static final int REGISTER = 1;
  ;; public static final int CHANGEOPS = 2;
  (hash-map :socket-channel socket-channel
            :type type
            :ops ops))

(defn- init-selector 
  "Create a new selector. Throws IOException."
  []
  (.openSelector (SelectorProvider/provider)))

(defn- initiate-connection
  ;; throws IOException 
  [client]
  ;; Create a non-blocking socket channel
  (let [sc (doto (SocketChannel/open)
             (.configureBlocking false)
             (.connect (:inet-address client)))] ;; Kick off connection establishment
    
    ;; Queue a channel registration since the caller is not the 
    ;; selecting thread. As part of the registration we'll register
    ;; an interest in connection events. These are raised when a channel
    ;; is ready to complete connection establishment.
    (swap! (:pending-changes client)
           conj 
           (change-request sc REGISTER SelectionKey/OP_CONNECT))
    sc))

(def q (clojure.lang.PersistentQueue/EMPTY))
(def q (conj q 1))
(def q (conj q 2))
(def q (conj q 3))

(empty? q)
(count q)
(into [] q)
(peek q)
(peek (pop q))
(peek (pop (pop (pop (pop q)))))

(conj [0 1] 2) ;; [0 1 2]
(peek [0 1 2]) ;; 2
(pop [0 1 2]) ;; [0 1]

(let [[f & r] [0 1 2]]
  r)

(conj '(1 0) 2)
(peek '(0 1 2)) ;; 0
(pop '(0 1 2)) ;; [1 2]

(defn- handle-response 
  ;; throws IOException 
  [client ^SocketChannel socket-channel data num-read]
  (let [rsp-data (byte-array num-read) ;; Make a correctly sized copy of the data before handing it to the client
        handler (get @(:rsp-handlers client) :socket-channel)] ;; Look up the handler for this channel
    (System/arraycopy data 0 rsp-data 0 num-read)   
    ;; pass the response to the handler
    (when (handler rsp-data)
      ;; The handler has seen enough, close the connection
      (.close socket-channel)
      (.cancel (.keyFor socket-channel (:selector client))))))

(defn- finish-connection
  ;; throws IOException
  [client ^SelectionKey key]
  (let [sc ^SocketChannel (.channel key)]
    ;; Finish the connection. If the connection operation failed
    ;; this will raise an IOException.
    (try (.finishConnect sc)
         ;; Register an interest in writing on this channel
         (.interestOps key SelectionKey/OP_WRITE)
         (catch IOException e
           ;; Cancel the channel's registration with our selector
           (.cancel key)))))




(defn nio-client [host port] ;;  The host:port combination to connect to
  {:inet-address (InetSocketAddress. host port)
   :selector (init-selector) ;; The selector we'll be monitoring
   :read-buf (ByteBuffer/allocate  8192) ;; The buffer into which we'll read data when it's available
   :pending-changes (atom (clojure.lang.PersistentQueue/EMPTY)) ;; A FIFO of pending changes
   :pending-data (atom {}) ;; Maps a SocketChannel to a list of ByteBuffer instances
   :rsp-handlers (atom {})} ;; Maps a SocketChannel to a RspHandler]))
  )


(defn send 
  ;; throws IOException
  [client data rsp-handler]
  (let [socket (initiate-connection client)] ;; Start a new connection
    ;; Register the response handler
    (swap! (:rsp-handlers client) assoc socket rsp-handler) 
    ;; Queue the data we want written
    (swap! (:pending-data client)
           update-in 
           [socket]
           (fnil #(swap! % conj (ByteBuffer/wrap data)) (atom (clojure.lang.PersistentQueue/EMPTY)))))
  (.wakeup ^Selector (:selector client)) ;; wake up our selecting thread so it can make the required changes
  )


(defn write 
  ;; throws IOException
  [client ^SelectionKey key]
  (let [sc (.channel key)
        pending-data (get @(:pending-data client) sc)]
    (loop [buf (peek @pending-data)]
      (when buf 
        (.write sc buf)
        (when (zero? (.remaining buf))
          (swap! pending-data pop)
          (recur))))
    (when (empty? @pending-data))
    ;; We wrote away all data, so we're no longer interested
    ;; in writing on this socket. Switch back to waiting for
    ;; data.
    (.interestOps key (SelectionKey/OP_READ))))

(defn read 
  ;; throws IOException
  [client ^SelectionKey selection-key]
  (let [sc (.channel selection-key)]
    (.clear (:read-buf client)) ;; Clear out our read buffer so it's ready for new data
    (try (let [num-read (.read sc (:read-buf client))] ;; Attempt to read off the channel
           (if (= -1 num-read)
             ;; Remote entity shut the socket down cleanly. Do the
             ;; same from our end and cancel the channel.
             (do (.close (.channel key))
                 (.cancel key)))
           ;;  Handle the response
           (handle-response sc (.array (:read-buf client) num-read))
           
           )
         (catch IOexception e
           ;; The remote forcibly closed the connection, cancel
           ;; the selection key and close the channel.
           (.cancel key)
           (.close sc)))))


(defn process-pending-changes 
  "Process pending changes and return updated value."
  [pc]
  (iterator-seq (.iterator pc))
  )

(defn event-loop-impl [client]
  (loop []

    (swap! (:pending-changes client) process-pending-changes)
    ;; Process any pending changes
    (locking (:pending-changes client)))
    
				synchronized (this.pendingChanges) {
					Iterator changes = this.pendingChanges.iterator();
					while (changes.hasNext()) {
						ChangeRequest change = (ChangeRequest) changes.next();
						switch (change.type) {
						case ChangeRequest.CHANGEOPS:
							SelectionKey key = change.socket.keyFor(this.selector);
							key.interestOps(change.ops);
							break;
						case ChangeRequest.REGISTER:
							change.socket.register(this.selector, change.ops);
							break;
						}
					}
					this.pendingChanges.clear();
				}


    ))


public class NioClient implements Runnable {
                                            


	public void run() {
		while (true) {
			try {

				// Wait for an event one of the registered channels
				this.selector.select();

				// Iterate over the set of keys for which events are available
				Iterator selectedKeys = this.selector.selectedKeys().iterator();
				while (selectedKeys.hasNext()) {
					SelectionKey key = (SelectionKey) selectedKeys.next();
					selectedKeys.remove();

					if (!key.isValid()) {
						continue;
					}

					// Check what event is available and deal with it
					if (key.isConnectable()) {
						this.finishConnection(key);
					} else if (key.isReadable()) {
						this.read(key);
					} else if (key.isWritable()) {
						this.write(key);
					}
				}
			} catch (Exception e) {
				e.printStackTrace();
			}
		}
	}






	public static void main(String[] args) {
		try {
			NioClient client = new NioClient(InetAddress.getByName("www.google.com"), 80);
			Thread t = new Thread(client);
			t.setDaemon(true);
			t.start();
			RspHandler handler = new RspHandler();
			client.send("GET / HTTP/1.0\r\n\r\n".getBytes(), handler);
			handler.waitForResponse();
		} catch (Exception e) {
			e.printStackTrace();
		}
	}
}
