(ns leiningen.gzip-resources
  (:require [clojure.java.io :as io])
  (:import (java.io File)
           (java.util.zip GZIPOutputStream)))

(defn regular-file? [^File file]
  (.isFile file))

(defn relativize ^File [^File file ^File base]
  (-> base
      .toURI
      (.relativize (-> file .toURI))
      .getPath
      io/file))

(defn target-file ^File [^File base ^File target ^File file]
  (io/file target (-> file
                      (.getAbsolutePath)
                      (str ".gz")
                      (io/file)
                      (relativize base))))

(defn mkdirs ^File [^File f]
  (-> f .getParentFile .mkdirs)
  f)

(defn length->str [length]
  (let [[length unit] (reduce (fn [[length unit] next-unit]
                                (if (<= length 1024.0)
                                  (reduced [length unit])
                                  [(/ length 1024.0) next-unit]))
                              [(double length) "bytes"]
                              ["kb" "mb" "gb"])]
    (format "%.1f %s" length unit)))

(defn gzip-file [abs-root-len [^File input ^File output]]
  (let [input-length (.length input)]
    (print (str "Gzipping "
                (-> input .getAbsolutePath (subs abs-root-len))
                ": "
                (length->str input-length)))
    (with-open [in  (-> input
                        (io/input-stream))
                out (-> output
                        (mkdirs)
                        (io/output-stream)
                        (GZIPOutputStream.))]
      (io/copy in out))
    (let [output-length (.length output)
          ratio         (/ (- input-length output-length) (double input-length))]
      (println (format " -> %s (%+.1f %%)"
                       (length->str output-length)
                       (* ratio -100.0))))))

(defn gzip-resources
  "Find and gzip resources. Generates an gzipped file of every resource. Generated
file name is same as the resource name with \".gz\" appended to it.

Accepts an optional map of following options:
  :target          - Target path, defaults to project :target
  :resource-paths  - Seq of resource paths, defaults to project :resource-paths"
  [project & [opts]]
  (let [target         (or (-> opts :target)
                           (-> project :target-path io/file))
        resource-paths (or (-> opts :resource-paths)
                           (-> project :resource-paths))
        base           (-> project :root (io/file))
        abs-root-len   (-> base .getAbsolutePath count)
        resources (->> resource-paths
                       (set)
                       (map io/file)
                       (mapcat file-seq)
                       (filter regular-file?)
                       (doall))]
    (->> resources
         (map (fn [file] [file (target-file base target file)]))
         (map (partial gzip-file abs-root-len))
         (dorun))))
