(ns blueprint.client
  (:require [aleph.http :as http]
            [blueprint.client.interceptors :as interceptors]
            [blueprint.client.spec-gen :as bsg]
            [blueprint.spec-gen :refer [normalize-path-param-name]]
            [blueprint.client.url :as url]
            [blueprint.core :as core]
            [blueprint.interceptor :as bpi]
            [blueprint.registry :as reg]
            [clojure.spec.alpha :as s]
            [exoscale.coax :as cx]
            [exoscale.ex :as ex]
            [exoscale.interceptor.manifold :as ixm]
            [exoscale.interceptor.protocols :as ixp]
            [manifold.executor :as me]
            [muuntaja.core :as m])
  (:import [java.util.concurrent ThreadPoolExecutor TimeUnit BlockingQueue LinkedBlockingQueue ThreadFactory]
           [java.util.concurrent.atomic AtomicInteger]))

(def default-interceptors
  [interceptors/execute-request
   interceptors/lens-response
   interceptors/decode
   interceptors/parse
   interceptors/get-response])

(defn- build-chain
  "Build an interceptor chain in three phases:

  - Start from the default chain
  - Process additional interceptors provided in `::additional`
  - Removes interceptors by name based on those provided in `::disabled`
  - Call the interceptor build step if any with the provided config"
  [config]
  (ex/assert-spec-valid ::interceptors config)
  (bpi/build-chain default-interceptors config))

(def build-with bpi/build-with)

;;
;; Threadpool Management
;;

(def ^:private default-thread-pool-size 64)

(defn- create-thread-factory
  "Creates a thread factory for HTTP client threads."
  [name-prefix]
  (let [counter (AtomicInteger. 0)]
    (reify ThreadFactory
      (newThread [_ runnable]
        (doto (Thread. runnable (str name-prefix "-" (.getAndIncrement counter)))
          (.setDaemon true))))))

(defn- create-limited-executor
  "Creates a limited ThreadPoolExecutor for HTTP requests."
  ([] (create-limited-executor "blueprint-http" default-thread-pool-size))
  ([pool-name pool-size]
   (ThreadPoolExecutor.
    (int pool-size)
    (int pool-size)
    60
    TimeUnit/SECONDS
    ^BlockingQueue (LinkedBlockingQueue.)
    ^ThreadFactory (create-thread-factory pool-name))))

(defn- create-connection-pool
  "Creates an aleph connection pool."
  [ssl-context]
  (if ssl-context
    (http/connection-pool {:connection-options {:ssl-context ssl-context}})
    http/default-connection-pool))

;;
;; Http Client
;;

(defn- base-url
  "Gets the base url for the defined api."
  [{:keys [servers] :as parsed-api}]
  (if-let [prefix (get-in parsed-api [:blueprint.options :path-prefix])]
    (str (:url (first servers)) prefix)
    (:url (first servers))))

(defn- resolve-server
  "Resolves the correct server according to the configured option."
  [str-or-ifn opts]
  (let [[t val] (s/conform :client-options/server str-or-ifn)]
    (cond (= t :str) val
          (= t :ifn) (val opts)
          :else (throw (ex-info (str "Unrecognized server option: " str-or-ifn) {:type t
                                                                                 :val val})))))

(defrecord Client [parsed-api command-defs server connection-pool interceptor-chain executor])

(defn make-client
  "Create a blueprint client from a blueprint api definition.
  
  Options:
  - `:ssl-context` - SSL context for HTTPS connections
  - `:server` - Override the server URL from the API definition
  - `:interceptor-chain` - Custom interceptor chain for the client
  - `:thread-pool-size` - Size of the HTTP client thread pool (default: 64)
  - `:thread-pool-name` - Name prefix for HTTP client threads (default: \"blueprint-http\")"
  ([api-def] (make-client api-def {}))
  ([api-def {:keys [ssl-context server interceptor-chain thread-pool-size thread-pool-name] :as default-options}]

   (ex/assert-spec-valid ::api-definition api-def)
   (ex/assert-spec-valid ::client-options default-options)

   (let [parsed-api (core/parse api-def)
         command-defs (bsg/apidef->commands-map parsed-api)
         pool-size (or thread-pool-size default-thread-pool-size)
         executor (create-limited-executor thread-pool-name pool-size)
         connection-pool (create-connection-pool ssl-context)
         server (or server (base-url parsed-api))]
     (->Client parsed-api command-defs server connection-pool interceptor-chain executor))))

(defn unstar-keys
  "Remove stars from key names for spec validation
  This is normally carried out by reitit router on the server
  because it's specific to its syntax"
  [path-params]
  (reduce-kv (fn [m k v]
               (assoc m (normalize-path-param-name k) v))
             {}
             path-params))

(def default-options
  {:pool-timeout 10e3
   :connection-timeout 10e3
   :request-timeout 10e3
   :read-timeout 10e3})

(defn invoke
  "Invoke a `command` on the remote blueprint-based api.
  Clients are created with `(blueprint.client/make-client api-def)`.
  Supported options in `opts`:

  | key                   | description |
  | ----------------------|-------------|
  | `:input`              | The request input to be sent
  | `:server`             | The remote server endpoint or a function (eg: \"http://localhost:8080\"); default is first entry from api definition
  | `:headers`            | A map of additional headers to be sent
  | `:options`            | A map of additional options to add to the request
  | `:as`                 | The format for content negotation; accepts `edn` or `json`; default: `json`
  | `:interceptors`       | The interceptor configuration to apply
  | `:interceptor-chain`  | The interceptor chain that will run the request; if specified `:interceptors` is not used 
  | `:throw-exceptions`   | Control throwing of exceptions in case of errors; default: `true`
  | `:decode-on-error`    | Control decoding of response body in case of errors; default: `true`
  "
  [{:keys [parsed-api command-defs] :as client}
   command
   {:keys [input
           server
           headers
           options
           as
           throw-exceptions
           decode-on-error
           interceptors
           interceptor-chain
           validate-input?
           coerce-input?
           initial-ctx]
    :or {input nil
         server (get client :server)
         headers {}
         as :json
         throw-exceptions true
         decode-on-error true
         validate-input? true
         coerce-input? true
         interceptors {}
         initial-ctx {}}
    :as opts}]

  (ex/assert-spec-valid ::client client)
  (ex/assert-spec-valid ::command command)
  (ex/assert-spec-valid ::options opts)
  (ex/assert-spec-valid ::interceptors interceptors)
  ;;ensure command is valid
  (ex/assert-spec-valid (into #{} (keys command-defs)) command {:message "Invalid command"})

  (let [command-def (get command-defs command)
        {::reg/keys [ns]} parsed-api
        {:as request-map :keys [url body query-params]} (url/cmd->request command command-def input)
        format (if (= as :edn)
                 "application/edn"
                 "application/json")

        input-spec (core/command-input-spec ns command)
        path-spec (core/command-path-spec ns command)
        params-spec (core/command-params-spec ns command)

        body (cond->> body coerce-input? (cx/coerce input-spec))
        input (cond->> input coerce-input? (cx/coerce path-spec))
        query-params (cond->> query-params coerce-input? (cx/coerce params-spec))]

    ;; validate command specifics
    (when validate-input?
      (ex/assert-spec-valid input-spec
                            body
                            {:message "Invalid input"})

      (ex/assert-spec-valid path-spec
                            (unstar-keys input)
                            {:message "Invalid input path params"})

      (ex/assert-spec-valid params-spec
                            query-params
                            {:message "Invalid input params"}))

    (let [target-url (str (resolve-server server opts) url)
          headers (cond-> (assoc headers "Accept" format)
                    body (assoc "Content-Type" format))
          base-req (-> request-map
                       (merge default-options options)
                       (assoc :url target-url
                              :headers headers
                              :throw-exceptions throw-exceptions
                              :decode-on-error decode-on-error
                              :pool (:connection-pool client)
                              :response-executor (:executor client)))
          http-req (cond-> base-req
                     body (assoc :body (m/encode format input))
                     query-params (assoc :query-params query-params))
          ;; chain is either local to this call, or fed via client instance, or
          ;; built on demand
          interceptor-chain (or interceptor-chain
                                (:interceptor-chain client)
                                (build-chain (assoc interceptors :command command-def)))]

      (ixm/execute (assoc initial-ctx
                          :request http-req
                          :blueprint.client/command-def command-def)
                   interceptor-chain))))

;;;;
;; Specs
;;;;

;; api-definition

(s/def ::api-definition :blueprint.core/definition)

;; client
(s/def ::client #(instance? Client %))

;; invoke arguments
(s/def ::command keyword?)

(s/def :client-options/ssl-context any?)
(s/def :client-options/server (s/or :str string? :ifn ifn?))
(s/def :client-options/thread-pool-size pos-int?)
(s/def :client-options/thread-pool-name string?)
(s/def ::client-options (s/nilable (s/keys :opt-un [:client-options/ssl-context
                                                    :client-options/server
                                                    :client-options/thread-pool-size
                                                    :client-options/thread-pool-name
                                                    ::interceptor-chain])))

;; config
(s/def ::input map?)
(s/def ::server string?)
(s/def ::as #{:json :edn})
(s/def ::throw-exceptions boolean?)
(s/def ::interceptors (s/keys :opt [::bpi/additional ::bpi/disabled]))
(s/def ::interceptor-chain (s/coll-of #(satisfies? ixp/Interceptor %)))
(s/def ::coerce-input? boolean?)
(s/def ::validate-input? boolean?)
(s/def ::options (s/nilable (s/keys :opt-un [::input ::server ::headers
                                             ::as ::throw-exceptions
                                             ::interceptors
                                             ::interceptor-chain
                                             ::coerce-input?
                                             ::validate-input?])))
