(ns blueprint.client.interceptors
  (:require [aleph.http :as http]
            [exoscale.interceptor :as ix]
            [clojure.tools.logging :as log]
            [muuntaja.core :as m]
            [clojure.spec.alpha :as s]
            [spec-tools.core :as st]))
;;
;; Private
;;
(defn- decodeable?
  "Indicates if the response can be decoded."
  [response]
  (some? (get-in response [:headers "content-type"])))

(defn- conform-body
  "Tries to conform the body according to the command's output spec.
  If decoding was unsuccessful, a `:invalid? true` keyval is added to the response, and the original
  body is left as is."
  [multispec response]
  (let [decoded  (st/decode multispec response st/json-transformer)
        invalid? (s/invalid? decoded)]
    (if invalid? (assoc response :invalid? true)
        decoded)))

(defn- decode-body
  "Decodes the http request body according to content type headers."
  [{:keys [headers] :as response}]
  ;; unfortunately, aleph http returns ring headers (lowercased)
  ;; but muuntaja expects Pascal-Case headers
  ;; so a hack is needed to ensure the m/decode-body-response finds the correct content-type
  ;; the alternative is to call m/decode-response-body with content-type and charset args
  ;; (m/decode response "application/edn" "utf-8"
  (let [norm-headers (assoc-in response [:headers "Content-Type"] (get headers "content-type"))]
    (->> norm-headers
         (m/decode-response-body)
         (assoc response :body))))

(defn- try-parse-body
  "Tries to manually parse the body. Does a best-effort to parse the body.
  If the body could not be parsed, returns the raw stream."
  [{:keys [request response] :as ctx}]
  ;; m/decode-response-body, via jackson, barfs if an empty input stream is decoded
  ;; https://github.com/FasterXML/jackson-databind/issues/907
  ;; it's hard to reliably check for an empty stream
  ;; (.available (:body response)) returns zero sometimes (when the stream is not yet :aleph/complete ?)
  ;; so we can't reliably set the body to nil when the response stream says available=0
  (try
    (let [format  (or (get-in request [:headers "Accept"])
                      (m/default-format m/instance))
          charset (m/default-charset m/instance)
          body    (:body response)
          decoded (m/decode m/instance format body charset)]
      (assoc-in ctx [:response :body] decoded))
    (catch Throwable e
      (log/error e "Could not parse response body")
      ctx)))

(def execute-request
  {:name  ::execute-request
   :enter (-> http/request
              (ix/in [:request])
              (ix/out [:response]))})

(def lens-response
  {:name  ::lens-response
   :enter (ix/lens identity [:response])})

(def get-response
  {:name  ::get-response
   :leave (ix/in identity [:response])})

(def parse
  {:name  ::parse-body
   :enter (-> try-parse-body
              (ix/when #(not (decodeable? (:response %)))))})

;; decoder requires a response-multispec
(s/def ::response-spec s/spec?)
(s/def ::decoder-config (s/keys :req-un [::response-spec]))
(def decode
  {:name    ::decode
   :spec    ::decoder-config
   :builder (fn build-decoder [interceptor {:keys [response-spec] :as config}]
              (let [conform-body-fn (partial conform-body response-spec)]
                (-> interceptor
                    (assoc :enter (-> (comp conform-body-fn decode-body)
                                      (ix/lens [:response])
                                      (ix/when #(decodeable? (:response %))))))))})
