(ns blueprint.spec-gen
  "A generator of clojure specs for data-based API definitions.

  This differs from spec-gen.clj by creating registered specs. So they
  are reachable by just using keywords with vanilla the spec api.

  * resource specs are in the form of `exoscale.test-api.resource/job`
  with for instance one of it's fields:

  `exoscale.test-api.resource.job/id` where `exoscale.test-api` is the
  blueprint definition root namespace, `job` the resource and `id` one
  of its fields.

  * commands specs for :update-job-posting are are under:
    - `exoscale.test-api.command.update-job-posting/output`
    - `exoscale.test-api.command.update-job-posting/input`
    - `exoscale.test-api.command.update-job-posting/params`
    - `exoscale.test-api.command.update-job-posting/path`
    - `exoscale.test-api.command.update-job-posting/handler`

  * resources
    - `exoscale.test-api.resource/job`
    - `exoscale.test-api.resource/user`"
  (:require [clojure.spec.alpha :as s]
            [clojure.string :as str]
            [exoscale.specs :as spec]
            [exoscale.specs.net :as net]
            [exoscale.specs.string :as st]
            [blueprint.registry :as reg]))

(defn resource-spec
  [ns k]
  (keyword (format "%s.resource"
                   (name ns))
           (name k)))

(defn command->key
  "Takes a whole keyword and generate a part good for `name` usage,
  including the ns potentially. :foo.bar/baz -> \"foo-bar-baz\""
  [k]
  (cond
    (or (simple-keyword? k)
        (string? k))
    (name k)

    (qualified-keyword? k)
    (format "%s.%s"
            (namespace k)
            (name k))))

(defn command-spec
  [ns k scope]
  (keyword (format "%s.command.%s"
                   (name ns)
                   (command->key k))
           scope))

(defn resource-ns
  [ns]
  (format "%s.resource" (name ns)))

(defn gen-spec!
  [spec-key spec-form meta]
  ;; prevent generating the same key over and over
  (when-not (s/get-spec spec-key)
    (eval `(-> (s/def ~spec-key ~spec-form)
               (spec/with-meta! '~meta)))))

(defn- spec-ref
  [spec-key]
  ;; workaround https://clojure.atlassian.net/browse/CLJ-2067
  (if (s/get-spec spec-key)
    spec-key
    `(s/and ~spec-key)))

(defmulti -compile-spec
  (fn [m _]
    (:type m)))

(defmethod -compile-spec :map
  [{:keys [attributes key-ns]} {:keys [ns] :as ctx}]
  (let [k-ns (or key-ns (gensym (format "%s$attr" (name ns))))]
    ;; first we need to generate all the keys specs, assuming they don't
    ;; exist yet
    (run! (fn [{:as attr :keys [attribute def]}]
            (gen-spec! (keyword (name k-ns)
                                (name attribute))
                       (-compile-spec (assoc def :ns ns)
                                      ctx)
                       attr))
          attributes)
    (letfn [(gen-xform-req [f]
              (comp (f (fn [{:as _attr :keys [opts]}]
                         (:req? opts)))
                    (map (fn [{:as _attr :keys [attribute]}]
                           (keyword (name k-ns)
                                    (name attribute))))))]
      `(s/keys :req-un ~(into []
                              (gen-xform-req filter)
                              attributes)
               :opt-un ~(into []
                              (gen-xform-req remove)
                              attributes)))))

(defmethod -compile-spec :coll-of
  [{:keys [def opts]} ctx]
  `(s/coll-of ~(-compile-spec def ctx)
              ~@(apply concat opts)))

(defmethod -compile-spec :set-of
  [{:keys [def]} ctx]
  `(s/coll-of ~(-compile-spec def ctx)
              :kind set?))

(defmethod -compile-spec :map-of
  [{:keys [key-def val-def]} ctx]
  `(s/map-of ~(-compile-spec key-def ctx)
             ~(-compile-spec val-def ctx)))

(defmethod -compile-spec :nilable
  [{:keys [def]} ctx]
  `(s/nilable ~(-compile-spec def ctx)))

(defmethod -compile-spec :string-of
  [{:keys [opts]} _ctx]
  `(st/string-of ~opts))

(defmethod -compile-spec :enum
  [{:keys [values]} _ctx]
  values)

(defmethod -compile-spec :reference
  [{:keys [reference]} {:keys [ns resource-ns]}]
  (spec-ref (keyword (name (or resource-ns ns))
                     (name reference))))

(defmethod -compile-spec :raw-json-schema
  [{:keys [raw-json-schema]} _]
  `map?)

(defmethod -compile-spec :spec
  [{:keys [spec]} _]
  (spec-ref spec))

(defmethod -compile-spec :pred
  [{:keys [pred]} _ctx]
  pred)

(defmethod -compile-spec :fn
  [{:keys [pred]} _ctx]
  pred)

(defmethod -compile-spec :symbol
  [{:keys [pred]} _ctx]
  pred)

(s/def ::ip
  (-> ::net/ip
      (spec/with-meta! {:description "IP address"
                        :reason "Invalid IP address"})))

(defmethod -compile-spec :ip
  [_ _]
  ::ip)

(s/def ::ipv4
  (-> ::net/ipv4
      (spec/with-meta! {:description "IPv4 address"
                        :reason "Invalid IPv4 address"})))
(defmethod -compile-spec :ipv4
  [_ _]
  ::ipv4)

(s/def ::ipv6
  (-> ::net/ipv6
      (spec/with-meta! {:description "IPv6 address"
                        :reason "Invalid IPv6 address"})))
(defmethod -compile-spec :ipv6
  [_ _]
  ::ipv6)

(defn- gen-num-spec [body]
  `(s/and number? ~body))

(defmethod -compile-spec :<
  [{:keys [floor ceiling]} _ctx]
  (gen-num-spec
   (if (some? ceiling)
     `(fn [x#] (< ~floor x# ~ceiling))
     `(fn [x#] (< x# ~floor)))))

(defmethod -compile-spec :<=
  [{:keys [floor ceiling]} _ctx]
  (gen-num-spec
   (if (some? ceiling)
     `(fn [x#] (<= ~floor x# ~ceiling))
     `(fn [x#] (<= x# ~floor)))))

(defmethod -compile-spec :>
  [{:keys [floor ceiling]} _ctx]
  (gen-num-spec
   (if (some? ceiling)
     `(fn [x#] (> ~floor x# ~ceiling))
     `(fn [x#] (> x# ~floor)))))

(defmethod -compile-spec :>=
  [{:keys [floor ceiling]} _ctx]
  (gen-num-spec
   (if (some? ceiling)
     `(fn [x#] (>= ~floor x# ~ceiling))
     `(fn [x#] (>= x# ~floor)))))

(defmethod -compile-spec :not=
  [{:keys [val]} _ctx]
  `(fn [x#] (not= ~val x#)))

(defn gen-multi-spec-defmulti!
  [mm-sym dispatch alternatives default ctx]
  (eval `(defmulti ~mm-sym ~dispatch))
  (doseq [{:keys [dispatch-val def]} alternatives]
    (eval `(defmethod ~mm-sym ~dispatch-val
             [_#]
             ~(-compile-spec def ctx))))
  (when default
    (eval `(defmethod ~mm-sym :default
             [arg#]
             (~default arg#)))))

(defmethod -compile-spec :multi
  [{:keys [dispatch alternatives default]} ctx]
  (let [mm-sym (gensym "multi-spec-mm")]
    (gen-multi-spec-defmulti! mm-sym
                              dispatch
                              alternatives
                              default
                              ctx)
    `(s/multi-spec ~mm-sym ~dispatch)))

(defmethod -compile-spec :default
  [x _]
  x)

(defn gen-spec-for-resource!
  [ns spec-key resource-def]
  (let [spec-ns (format "%s.resource" (name ns))
        spec-name (keyword spec-ns (name spec-key))]
    (gen-spec! spec-name
               (-compile-spec (assoc resource-def
                                     :key-ns (format "%s.resource.%s"
                                                     (name ns)
                                                     (name spec-key)))
                              {:ns spec-ns
                               :resource-ns (resource-ns ns)})

               resource-def)))

(defn gen-resources-specs!
  [{::reg/keys [resources ns]}]
  (run! (fn [[resource-key resource-def]]
          (gen-spec-for-resource! ns
                                  resource-key
                                  resource-def))
        resources))

(defn normalize-path-param-name
  "Normalizes catch-all syntax path parameters
  *foo -> foo"
  [kw-name]
  (let [str-name (name kw-name)]
    (if (str/starts-with? str-name "*")
      (keyword (subs str-name 1))
      kw-name)))

(defn gen-commands-specs!
  [{::reg/keys [commands ns]}]
  (doseq [[command-key {:as command
                        :keys [output params input path]}] commands
          :let [spec-ns (format "%s.command.%s"
                                (name ns)
                                (command->key command-key))]]
    ;; output specs
    (let [spec-name (keyword spec-ns "output")
          spec-root (str spec-ns ".output")]
      (gen-spec! spec-name
                 (if output
                   (-compile-spec {:type :multi
                                   :dispatch :status
                                   :alternatives (mapv (fn [[k v]]
                                                         {:def {:type :map
                                                                :attributes
                                                                [{:attribute :body
                                                                  :def v}]}
                                                          :dispatch-val k})
                                                       output)
                                   :default `(complement any?)}
                                  {:ns spec-root
                                   :resource-ns (resource-ns ns)})
                   `any?)
                 command))

    ;; params specs
    (let [spec-name (keyword spec-ns "params")
          spec-root (str spec-ns ".params")]
      (gen-spec! spec-name
                 (if params
                   (-compile-spec {:type :map
                                   :attributes (mapv (fn [[name spec]]
                                                       {:attribute name :def spec})
                                                     params)}
                                  {:ns spec-root
                                   :resource-ns (resource-ns ns)})
                   `any?)
                 command))

    ;; inputs specs
    (let [spec-name (keyword spec-ns "input")
          spec-root (str spec-ns ".input")]
      (gen-spec! spec-name
                 (if input
                   (-compile-spec input
                                  {:ns spec-root
                                   :resource-ns (resource-ns ns)})
                   `any?)
                 command))

    ;; path specs
    (let [spec-name (keyword spec-ns "path")
          spec-root (str spec-ns ".path")
          args (keep (fn [[k & val]]
                       (when (= k :arg)
                         (first val)))
                     (:elems path))]
      (gen-spec! spec-name
                 (-compile-spec
                  (if (seq args)
                    {:type :map
                     :attributes
                     (map (fn [{:keys [name def]}]
                            {:attribute (normalize-path-param-name name)
                             :def def
                             :opts {:req? true}})
                          args)}
                    `any?)
                  {:ns spec-root
                   :resource-ns (resource-ns ns)})
                 command))

    ;; handler spec (merge of all the above (without output obviously))
    (let [spec-name (keyword spec-ns "handler")
          spec-root (str spec-ns ".handler")]
      (gen-spec! spec-name
                 `(s/merge ~@[(keyword spec-ns "input")
                              (keyword spec-ns "path")
                              (keyword spec-ns "params")])
                 {:ns spec-root
                  :resource-ns (resource-ns ns)}))))

;; Build a map of operation name to spec, and add an `handler` key for
;; a multi-spec dispatch. We can memoize it since we assume schemas are
;; static
(defn generate-specs!
  [api-def]
  (doto api-def
    gen-resources-specs!
    gen-commands-specs!))

(def compile-spec #'-compile-spec)
