(ns blueprint.spec
  (:require [clojure.spec.alpha  :as s]))

(s/def ::builtin
  (s/or
   :symbol
   #{'uuid? 'pos-int? 'string? 'boolean? 'int?
     'set? 'map? 'any? 'vector? 'list? 'inst?
     'keyword? 'number?}
   :fn
   #{uuid? pos-int? string? boolean?
     set? map? any? vector? list? inst?
     keyword? number?}))

(s/def ::desc string?)
(s/def ::ro?  boolean?)
(s/def ::req? boolean?)
(s/def ::visible? boolean?)
(s/def ::auth? boolean?)
(s/def ::map-def-opts map?)

;; each extension is a value of the map that can be anything, including nil
(s/def ::extensions map?)

(s/def ::attribute-opts
  (s/and map?
         (s/keys :req-un [::desc] :opt-un [::ro? ::req?])))

(s/def ::map-def map?)

(s/def ::vector-type
  #{:map :multi :and :coll-of :map-of :or-fields :enum
    :> :< :>= :<= :not=})

(s/def ::vector-def
  (s/or
   :map
   (s/cat :type       #{:map}
          :opts       (s/? ::map-def-opts)
          :attributes (s/+ (s/and vector?
                                  (s/cat :attribute keyword?
                                         :opts (s/? ::attribute-opts)
                                         :def ::resource-def))))

   :multi
   (s/cat :type         #{:multi}
          :dispatch     any?
          :alternatives (s/+ (s/and vector?
                                    (s/cat :dispatch-val any?
                                           :def ::resource-def))))

   :and
   (s/cat :type #{:and}
          :opts (s/? ::map-def-opts)
          :defs (s/+ ::resource-def))

   :coll-of
   (s/cat :type #{:coll-of}
          :opts (s/? ::map-def-opts)
          :def  ::resource-def)

   :map-of
   (s/cat :type #{:map-of}
          :opts (s/? ::map-def-opts)
          :key-def  ::resource-def
          :val-def  ::resource-def)

   :or-fields
   (s/cat :type   #{:or-fields}
          :fields (s/+ keyword?))

   :enum
   (s/cat :type   #{:enum}
          :fields (s/+ (s/or :string string? :keyword keyword?)))

   :int-condition
   (s/cat :type #{:>= :<= :> :< :not=}
          :floor number?
          :ceiling (s/? number?))))

(s/def ::resource-def (s/or :enum set?
                            :ip #{:ip}
                            :ipv4 #{:ipv4}
                            :ipv6 #{:ipv6}
                            :reference keyword?
                            :builtin ::builtin
                            :vector ::vector-def))

(s/def ::path-arg
  (s/cat :name keyword?
         :def  ::resource-def))

(s/def ::path (s/cat :method #{:get :post :delete :put :patch}
                     :elems (s/+ (s/or :string string?
                                       :arg ::path-arg))
                     :extensions (s/? ::extensions)))
(s/def ::output-opts (s/keys :opt-un [::desc]))
(s/def ::output (s/map-of (s/or :code (s/and pos-int? #(<= 200 % 600))
                                :default #{:default})
                          ::resource-def))
(s/def ::output-extensions ::extensions)
(s/def ::input ::resource-def)
(s/def ::params (s/map-of keyword? ::resource-def))
(s/def ::options (s/keys :opt-un [::visible? ::auth?]))
(s/def ::summary string?)
(s/def ::command-def (s/keys :req-un [::path (or ::desc ::summary) ::output]
                             :opt-un [::input ::params ::options ::extensions ::output-extensions]))

(s/def ::resources (s/map-of keyword? ::resource-def))
(s/def ::commands (s/map-of keyword? ::command-def))


;; Info


(s/def ::version     string?)
(s/def ::description string?)
(s/def ::title       string?)
(s/def ::info        (s/keys :req-un [::description ::version ::title]))

(defn documented-variables?
  [{:keys [url variables]}]
  (let [m  (re-matcher #"\{([^}]*)\}" url)
        ks (set (keys variables))]
    (empty?
     (for [_      (range)
           :let   [[_ v] (re-find m)]
           :while (some? v)
           :when  (not (contains? ks (keyword v)))]
       v))))

(s/def ::url     string?)
(s/def ::enum (s/coll-of string?))
(s/def ::default string?)
(s/def ::variables (s/map-of keyword? (s/keys :req-un [::default ::enum])))
(s/def ::server  (s/and
                  (s/keys :req-un [::url]
                          :opt-un [::variables ::description])
                  documented-variables?))
(s/def ::servers (s/coll-of ::server))

;; Tags
(s/def ::intro string?)
(s/def ::external-docs (s/keys :req-un [::url ::description]))
(s/def ::tag-def (s/keys :req-un [::description ::external-docs]))
(s/def ::tags (s/map-of keyword? ::tag-def))

;; Base schema
(s/def :blueprint.registry/blueprint
  (s/keys :req-un [::commands ::resources ::tags ::info ::servers]))

(s/def :blueprint.core/blueprint :blueprint.registry/blueprint)
(s/def :blueprint.core/definition :blueprint.registry/blueprint)
