(ns blueprint.core
  (:require [clojure.edn          :as edn]
            [clojure.java.io      :as io]
            [clojure.spec.alpha   :as s]
            [exoscale.ex          :as ex]
            [spec-tools.core      :as st]
            [spec-tools.data-spec :as ds]
            [blueprint.registry   :as reg]
            [blueprint.spec-gen   :as sg]
            [blueprint.spec       :as bs]))

(defn parse
  [source]
  (-> source
      (reg/build)
      (sg/generate-specs)
      (assoc :parsed? true)))

(defn spec-for
  [api-def k]
  (-> api-def :specs (get k)))

(defn output-spec-for
  [api-def k]
  (-> api-def :output-specs (get k)))

(def resource-spec #'spec-for)
(def command-spec #'spec-for)

(defn handler-spec
  [api-def]
  (spec-for api-def :handler))

(defn forwarded-fn
  "Creates a function which behaves like its namesake in clojure.spec.alpha.
   The created function may take an additional first argument containing a
   blueprint registry if references are used."
  [forwarded]
  (fn internal-forwarded-fn
    ([spec input]
     (internal-forwarded-fn {} spec input))
    ([registry spec input]
     (ex/assert-spec-valid ::bs/resource-def spec)
     (binding [sg/*resources* registry]
       (forwarded (-> (s/conform ::bs/resource-def spec)
                      reg/clean-resource
                      sg/compile-spec)
                  input)))))

(defmacro def-spec-forward
  "Create a forwarded function as for `forwarded-fn` and bind it to the given
   symbol. The target of the forwarded function is the namesake of the provided
   symbol `sym`."
  [sym]
  (let [spec-sym (symbol "clojure.spec.alpha" (name sym))]
    `(alter-meta!
      (def ~sym ~(str "Acts as " spec-sym " for a blueprint spec and registry")
        (forwarded-fn ~spec-sym))
      assoc :arglists '([~'spec ~'input] [~'registry ~'spec ~'input]))))

;; Forward a few common ones
(def-spec-forward valid?)
(def-spec-forward explain)
(def-spec-forward explain-data)
(def-spec-forward explain-str)
(def-spec-forward conform)
(def-spec-forward assert*)
