(ns blueprint.registry
  "A data-based registry of specifications for input and commands
   in an HTTP API."
  (:require blueprint.spec
            [clojure.spec.alpha :as s]
            [exoscale.ex        :as ex]))

(defmulti clean-resource
  (fn [[type v]]
    (let [res (if (= type :vector) (first v) type)]
      res)))

(defmethod clean-resource :map
  [[_vector [_map {:keys [attributes] :as m}] _mapvec]]
  (assoc m :attributes
         (vec (for [attr attributes]
                (update attr :def clean-resource)))))

(defmethod clean-resource :map-of
  [[_vector [_map-of m]]]
  (-> m
      (update :key-def clean-resource)
      (update :val-def clean-resource)))

(defmethod clean-resource :nilable
  [[_vector [_nilable m]]]
  (update m :def clean-resource))

(defmethod clean-resource :coll-of
  [[_vector [_coll-of m]]]
  (-> m
      (update :def clean-resource)
      (update :opts select-keys [:distinct :min-count :max-count :count])))

(defmethod clean-resource :set-of
  [[_vector [_set-of m]]]
  (update m :def clean-resource))

(defmethod clean-resource :string-of
  [[_vector [_string-of m]]]
  m)

(defmethod clean-resource :or-fields
  [[_vector [_or-fields m]]]
  m)

(defmethod clean-resource :int-condition
  [[_vector [_int-cond m]]]
  m)

(defmethod clean-resource :> [[_ [_ floor ceiling]]]  [floor ceiling])
(defmethod clean-resource :>= [[_ [_ floor ceiling]]] [floor ceiling])
(defmethod clean-resource :<= [[_ [_ floor ceiling]]] [floor ceiling])
(defmethod clean-resource :< [[_ [_ floor ceiling]]] [floor ceiling])
(defmethod clean-resource :not= [[_ [_ m]]] m)

(defmethod clean-resource :spec
  [[_ spec]]
  {:type :spec :spec spec})

(defmethod clean-resource :raw-json-schema
  [[_ [_ raw-json-schema]]]
  {:type :raw-json-schema
   :raw-json-schema (assoc-in raw-json-schema [:opts :type] "object")})

(defmethod clean-resource :reference
  [[_ reference]]
  {:type :reference :reference reference})

(defmethod clean-resource :ip
  [_]
  {:type :ip})

(defmethod clean-resource :ipv4
  [_]
  {:type :ipv4})

(defmethod clean-resource :ipv6
  [_]
  {:type :ipv6})

(defmethod clean-resource :enum
  [[_ values]]
  {:type :enum :values values})

(defmethod clean-resource :builtin
  [[_ x]]
  {:type :pred :pred (cond
                       (symbol? x)
                       (deref (ns-resolve 'clojure.core x))

                       (vector? x)
                       (second x)

                       :else
                       x)})

(defn clean-entry
  [m k v]
  (assoc m k (clean-resource v)))

(defn clean-command
  [m k {:keys [input params] :as cmd}]
  (assoc m
         k
         (cond-> (-> cmd
                     (update :output (partial reduce-kv clean-entry {}))
                     (update-in [:path :elems]
                                #(for [[t def] %]
                                   (if (= t :arg)
                                     [:arg {:name (:name def)
                                            :def  (clean-resource (:def def))}]
                                     [t def]))))

           (some? input)
           (update :input clean-resource)
           (some? params)
           (update :params (partial reduce-kv clean-entry {})))))

(def default-ns 'blueprint.spec-gen.api)

(defn build
  [source]
  (ex/assert-spec-valid ::blueprint source)
  (let [{:keys [resources commands ns]} (s/conform ::blueprint source)]
    (assoc source
           ::resources (reduce-kv clean-entry {} resources)
           ::commands  (reduce-kv clean-command {} commands)
           ::ns (or ns default-ns))))
