(ns blueprint.error
  (:require [exoscale.ex :as ex]
            [clojure.tools.logging :as log]))

(defn ex->response*
  "Format user-exposable exceptions"
  [e defaults]
  (let [status (:http/status defaults 500)]
    {:status status
     :body (ex-message e)}))

(defmulti ex->response
  "Controls how exception response will happen.
  It will dispatch the call to the defmethods via the
  exception `:type`, defaulting to a generic response."
  (fn [e]
    (some-> e ex-data :type))
  :hierarchy exoscale.ex/hierarchy)

(defmethod ex->response ::ex/incorrect [e] (ex->response* e {:http/status 431}))
(defmethod ex->response ::ex/not-found [e] (ex->response* e {:http/status 404}))
(defmethod ex->response ::ex/fault [e] (ex->response* e {:http/status 500}))
(defmethod ex->response ::ex/forbidden [e] (ex->response* e {:http/status 403}))
(defmethod ex->response ::ex/interrupted [e] (ex->response* e {:http/status 206}))
(defmethod ex->response ::ex/unsupported [e] (ex->response* e {:http/status 405}))
(defmethod ex->response ::ex/conflict [e] (ex->response* e {:http/status 409}))
(defmethod ex->response ::ex/unavailable [e] (ex->response* e {:http/status 504}))
(defmethod ex->response ::ex/busy [e] (ex->response* e {:http/status 503}))

(defn unexposable-handler
  "Install the method for the ::unexposable errors."
  [default-error-msg]
  (defmethod ex->response ::ex/unexposable [_]
    {:body default-error-msg
     :status 500}))

(prefer-method ex->response ::ex/unexposable ::ex/incorrect)
(prefer-method ex->response ::ex/unexposable ::ex/not-found)
(prefer-method ex->response ::ex/unexposable ::ex/fault)
(prefer-method ex->response ::ex/unexposable ::ex/forbidden)
(prefer-method ex->response ::ex/unexposable ::ex/interrupted)
(prefer-method ex->response ::ex/unexposable ::ex/unsupported)
(prefer-method ex->response ::ex/unexposable ::ex/conflict)
(prefer-method ex->response ::ex/unexposable ::ex/unavailable)
(prefer-method ex->response ::ex/unexposable ::ex/busy)

;; That would only be called if exception descends from ::ex/user-exposable
(defmethod ex->response :default [ex]
  (ex->response* ex {:http/status 500}))

(defmulti ex->log
  "Controls how exception logging will happen if at all.
  It will dispatch the call to the defmethods via the
  exception `:type`, defaulting to generic logging."
  (fn [e ctx]
    (some-> e ex-data :type))
  :hierarchy exoscale.ex/hierarchy)

(defn find-message
  "Finds most appropriate message within exception stack for
  logging/reporting purposes. Tries to avoid the default
  `default-error-msg`"
  [e default-error-msg]
  (let [msg (ex-message e)]
    (if (and (some? msg)
             (not= default-error-msg msg))
      msg
      (or (some-> e ex-cause find-message)
          default-error-msg))))

(defmethod ex->log :exoscale.ex/no-logging [_ _] nil)

(defmethod ex->log :default
  [e ctx]
  (let [request-id (:request-id ctx)]
    (log/error e
               request-id
               (find-message e (:default-error-msg ctx))
               (pr-str (ex-data e)))))

;; interceptors

(def exception-logger
  {:name ::exception-logger
   :error (fn [ctx e]
            (ex->log e ctx)
            (throw e))})

(def exception-response
  {:name ::exception-response
   :error (fn [ctx e]
            (assoc ctx :response (ex->response e)))})
