(ns blueprint.handler.backward-compat
  (:require [aleph.http.params             :as params]
            [blueprint.handler.error       :as error]
            [blueprint.registry            :as reg]
            [clojure.tools.logging         :as log]
            [muuntaja.core                 :as m]
            [jsonista.core                 :as json]
            [exoscale.interceptor.manifold :as ixm]
            [exoscale.interceptor          :as interceptor]
            [expound.alpha                 :as expound]
            [spec-tools.core               :as st]
            [manifold.deferred             :as d]
            [clojure.spec.alpha            :as s]
            [exoscale.ex                   :as ex]))

(defn normalize
  [{:keys [get-params body-params path-params handler] :as request}]
  (merge (select-keys request [:blueprint.router/options])
         get-params
         path-params
         body-params
         {:handler handler}))

(defn ex-invalid-spec
  [spec input]
  (ex/ex-info (format "Invalid input parameters: %s" (s/explain-str spec input))
              [::invalid-spec [::ex/invalid-spec ::ex/user-exposable]]
              (s/explain-data spec input)))

(defn decode-input
  [spec input]
  (let [decoded (st/decode spec input st/json-transformer)]
    (when (= decoded ::s/invalid)
      (binding [st/*transformer* st/json-transformer
                st/*encode?*     false]
        (log/debugf "Input does not conform: %s"
          (with-out-str (st/explain spec input st/json-transformer)))
        (throw (ex-invalid-spec spec input))))
    decoded))

(def muuntaja-errors
  "An exhaustive list of error types thrown by the muntaaja library"
  #{:muuntaja/decode
    :muuntaja/request-charset-negotiation
    :muuntaja/response-charset-negotiation
    :muuntaja/response-format-negotiation})

(defn wrap-incorrect
  "A function which will wrap an exception with an exception that derives
  ::ex/incorrect and ::ex/user-exposable, while copying messages and ex-data"
  [cause]
  (ex/ex-info (ex-message cause)
              [::incorrect [::ex/incorrect ::ex/user-exposable]]
              (ex-data cause)
              cause))

(defn interceptor-map
  [api-def handler]
  (let [router (:router api-def)
        specs  (::reg/specs api-def)]

    {::request-id
     {:name  :request-id
      :enter #(assoc % :request-id (java.util.UUID/randomUUID))}

     ::final
     {:name  :final
      :leave :response}

     ::format
     {:name  :format
      :enter (fn [{:keys [request] :as context}]
               (let [request (m/negotiate-and-format-request m/instance request)]
                 (assoc context ::request request :request request)))

      :leave (-> (fn [{::keys [request] :keys [response] :as context}]
                   (m/format-response m/instance request response))
                 (interceptor/out [:response]))

      :error (fn [_ e] (throw (if (contains? muuntaja-errors (-> e ex-data :type))
                                (wrap-incorrect e)
                                e)))}
     ::params
     params/interceptor

     ::route
     {:name  :route
      :enter (-> router
                 (interceptor/lens [:request]))}

     ::normalize
     {:name  :normalize
      :enter (-> normalize
                 (interceptor/lens [:request]))}

     ::transform
     {:name  :transform
      :enter (-> (partial decode-input (:handler specs))
                 (interceptor/lens [:request]))}

     ::handler
     {:name  :handler
      :enter (-> handler
                 (interceptor/in [:request])
                 (interceptor/out [:response]))}

     ;; Not pulled in by default
     ;; Allows handlers to either send back a body directly or
     ;; it will be wrapped in it
     ::wrap-response-body
     {:name  :wrap-response-body
      :leave (fn [{:keys [response] :as ctx}]
               (if (and (map? response) (contains? response :body))
                 ctx
                 (assoc ctx :response {:body response})))}

     ::error-response
     error/interceptor

     ::error-logger
     {}}))

(def default-interceptor-chain
  [::final
   ::error-response
   ::request-id
   ::format
   ::params
   ::route
   ::normalize
   ::transform
   ::wrap-response-body
   ::handler])

(defn generate-ring-handler
  ([api-def handler] (generate-ring-handler api-def handler {}))
  ([api-def handler {:keys [user-interceptors
                            interceptor-chain
                            default-error-msg]
                     :or {default-error-msg "Internal error"
                          interceptor-chain default-interceptor-chain}}]
   (let [interceptors (merge (interceptor-map api-def handler) user-interceptors)
         chain ((apply juxt interceptor-chain) interceptors)]
     (fn [request]
       (d/catch (ixm/execute {:request request
                              :default-error-msg default-error-msg}
                             chain)
                (fn [x]
                  (log/error x "unhandled http error")
                  {:status 500
                   :body   default-error-msg}))))))
