(ns blueprint.handler
  "Facilities to build an asynchronous handler, compatible with
   aleph, out of an interceptor chain.

   This builds an opinionated chain with sane defaults, and a way
   to adapt the chain to the consumer's liking.

   Processing of the chain is done by
   [interceptor](https://github.com/exoscale/interceptor)."
  (:require [blueprint.handler.backward-compat :as bc]
            [blueprint.handler.request-id      :as request-id]
            [blueprint.handler.default         :as default]
            [blueprint.handler.error           :as error]
            [blueprint.handler.format          :as format]
            [aleph.http.params                 :as params]
            [exoscale.interceptor.manifold     :as ixm]
            [exoscale.ex                       :as ex]
            [clojure.spec.alpha                :as s]
            [blueprint.core                    :as bp]))

(def default-interceptors
  "The default interceptor chain. doc/interceptor.md
   should be updated when this changes."
  [error/last-ditch             ;; A last-ditch error catcher
   default/final                ;; Extract :response out
   request-id/interceptor       ;; Add request ID
   format/enter                 ;; Content negotiation and (de)serialization
   format/leave
   error/interceptor            ;; Catch/log errors
   params/interceptor
   default/route
   default/normalize
   default/transform
   default/wrap-response-body
   default/handler])

(defn build-with
  "Interceptor builder, if a `:builder` key is found in the
   interceptor, it will be called on itself, with the configuration
   as a second argument. Validates with spec at `:spec` key if provided.

   This allows providing configuration to the chain"
  [config {:keys [builder spec] :as interceptor}]
  (when spec (ex/assert-spec-valid spec config))
  (cond-> interceptor
    (some? builder) (builder config)))

(defn- insert-at
  "Inserts an element before or after another one in a chain"
  [pos chain target ix]
  (let [[head tail] (split-with #(not= target (:name %)) chain)]
    (cond
      (empty? tail)  (throw (ex/ex-info "no such target" ::ex/incorrect))
      (= pos :after) (concat head [(first tail) ix] (rest tail))
      :else          (concat head [ix] tail))))

(defn add-to-chain
  "Adds an interceptor to the chain, `:blueprint.handler/position` determines
  the position in the chain, and can be:

  - `:first`: Puts the interceptor at the beginning of the chain.
  - `:last`: Puts the interceptor at the end of the chain.
  - `:before`: Puts the interceptor before the one whose name is stored at the
    `:blueprint.handler/target` key in the input map.
  - `:after`: Puts the interceptor after the one whose name is stored at the
    `:blueprint.handler/target` key in the input map.

  Incorrect parameters will yield to thrown exceptions."
  [chain {::keys [position target] :as interceptor}]
  (vec
   (case position
     :last   (conj chain interceptor)
     :first  (into [interceptor] chain)
     :before (insert-at :before chain target interceptor)
     :after  (insert-at :after chain target interceptor)
     (throw (ex/ex-info "invalid position" ::ex/incorrect)))))

(defn place
  "Helper function to populate an interceptor with position info"
  ([interceptor position]
   (place interceptor position nil))
  ([interceptor position target]
   (cond-> (assoc interceptor ::position position)
     (some? target)
     (assoc ::target target))))

(defn build-chain
  "Build an interceptor chain in three phases:

  - Start from the default chain
  - Process additional interceptors provided in `::additional`
  - Removes interceptors by name based on those provided in `::disabled`
  - Call the interceptor build step if any with the provided config"
  [{::keys [additional disabled] :as config}]
  (ex/assert-spec-valid ::config config)
  (->> (reduce add-to-chain default-interceptors additional)
       (remove #(contains? (set disabled) (:name %)))
       (map (partial build-with config))
       (doall)))

(defn ring-handler
  "Build a ring handler from a configuration map, optionally
   takes the two required arguments: the API definition and
   handler function as arguments.

   Yields a function of a single request argument which
   processes the chain and yields a final deferred ring
   response"
  ([config]
   (ex/assert-spec-valid ::config config)
   (let [chain (build-chain config)]
     (fn [request] (ixm/execute {:request request} chain))))
  ([definition handler]
   (ring-handler definition handler nil))
  ([definition handler config]
   (ring-handler (assoc config ::bp/definition definition ::handler handler))))

(s/def ::disabled (s/coll-of qualified-keyword?))
(s/def ::name keyword?)
(s/def ::position #{:first :last :before :after})
(s/def ::target keyword?)
(s/def ::named-interceptor (s/keys :req-un [::name] :opt [::position ::target]))
(s/def ::additional (s/coll-of ::named-interceptor))
(s/def ::handler ifn?)
(s/def ::config (s/keys :req [::bp/definition ::handler]
                        :opt [::additional ::disabled]))

;; Forwarded for backward compatibility
;; ====================================

(def generate-ring-handler     #'bc/generate-ring-handler)
(def interceptor-map           #'bc/interceptor-map)
(def default-interceptor-chain #'bc/default-interceptor-chain)
