(ns blueprint.handler.default
  "Implementation of the default interceptors.
   See `blueprint.handler` for reasoning"
  (:require [clojure.spec.alpha    :as s]
            [spec-tools.core       :as st]
            [exoscale.ex           :as ex]
            [exoscale.interceptor  :as interceptor]
            [clojure.tools.logging :as log]
            blueprint.core))

(def final
  "An interceptor which extracts the response key to conform to ring behavior"
  {:name  :blueprint.handler/final
   :leave :response})

(def route
  "Routing interceptor, needs router from `blueprint.router/generate-router`"
  {:name    :blueprint.handler/route
   :spec    ::definition
   :builder (fn [this {:blueprint.core/keys [definition]}]
              (assoc this :enter (interceptor/lens (:router definition)
                                                   [:request])))})


(def not-found
  "Quit the chain when the handler is not found."
  {:name :blueprint.handler/not-found
   :enter
   (fn [ctx]
     (let [handler (get-in ctx [:request :handler])]
       (when (identical? handler :blueprint.core/not-found)
         (ex/ex-not-found! "Not found"))
       ctx))})


(def normalize
  "Request normalizer, merges params coming from the body,
   path, or query string and produces a single input map.
   Assoc's the handler name to allow command spec evaluation"
  {:name  :blueprint.handler/normalize
   :enter (-> (fn [{:keys [get-params body-params path-params handler]
                    :as request}]
                (merge (select-keys request [:blueprint.router/options])
                       get-params
                       path-params
                       body-params
                       {:handler handler}))
              (interceptor/lens [:request]))})

(def add-original-request
  "Store original request"
  {:name  :blueprint.handler/original-request
   :enter #(assoc % ::original-request (:request %))})

(def original-request
  "Retrieve original request from a context"
  ::original-request)

(defn decode
  "Returns the value passed as parameter is the value is
  valid - otherwise, calls [[conform]] & [[unform]]. Returns `::s/invalid`
  if the value can't be decoded to conform the spec."
  ([spec coerced]
   (decode spec coerced nil))
  ([spec coerced transformer]
   (if (s/valid? spec coerced)
     coerced
     (binding [st/*transformer* transformer, st/*encode?* false]
       (let [conformed (s/conform spec coerced)]
         (if (s/invalid? conformed)
           conformed
           (s/unform spec conformed)))))))

(defn ^:no-doc transform-input
  [spec input]
  (let [coerced (st/coerce spec input st/json-transformer)
        decoded (decode spec coerced st/json-transformer)]
    (when (= decoded ::s/invalid)
      (binding [st/*transformer* st/json-transformer
                st/*encode?*     false]
        (log/debugf "Input does not conform: %s"
          (with-out-str (st/explain spec coerced st/json-transformer)))
        (throw (ex/ex-invalid-spec spec coerced))))
    decoded))

(def transform
  "Validates and coerces normalized input as per spec if possible or throws"
  {:name    :blueprint.handler/transform
   :spec    ::definition
   :builder (fn [this {:blueprint.core/keys [definition]}]
              (let [spec (get-in definition [:specs :handler])]
                (assoc this :enter (-> (partial transform-input spec)
                                       (interceptor/lens [:request])))))})

(def wrap-response-body
  "Wrap response from handler in a valid ring response map if none is found"
  {:name  :blueprint.handler/wrap-response-body
   :leave (interceptor/lens #(cond->> %
                               (not (and (map? %) (contains? % :body)))
                               (array-map :body))
                            [:response])})

(def handler
  "An interceptor which defers to a provided handler"
  {:name    :blueprint.handler/handler
   :spec    ::handler
   :builder (fn [this {:blueprint.handler/keys [handler]}]
              (assoc this :enter (-> handler
                                     (interceptor/in [:request])
                                     (interceptor/out [:response]))))})

(defn- log-request
  "Logs the request, including execution for response"
  [{:keys [::logger-ts ::original-request response] :as ctx}]
  (let [elapsed (- (System/currentTimeMillis) logger-ts)
        {:keys [server-name uri body scheme request-method]} original-request
        {:keys [status]} response]
   ;; sometimes status is null, I think aleph emits 200 in this case
   (log/infof "%s %s status %s [%sms]" request-method uri status elapsed)
   (dissoc ctx ::logger-ts)))

(def logger
  "A generic request/response/error logger that logs request and elapsed time.
  Rethrows any error"
  {:name  :blueprint.handler/logger
   :enter (fn [ctx]   (assoc ctx ::logger-ts (System/currentTimeMillis)))
   :leave (fn [ctx]   (log-request ctx) ctx)
   :error (fn [ctx e] (log-request ctx) (throw e))})

(s/def ::definition (s/keys :req [:blueprint.core/definition]))
(s/def ::handler    (s/keys :req [:blueprint.handler/handler]))
