(ns blueprint.handler.default
  "Implementation of the default interceptors.
   See `blueprint.handler` for reasoning"
  (:require [clojure.spec.alpha    :as s]
            [exoscale.ex           :as ex]
            [exoscale.coax         :as cx]
            [exoscale.interceptor  :as interceptor]
            [clojure.tools.logging :as log]
            [blueprint.core        :as core]))

(def final
  "An interceptor which extracts the response key to conform to ring behavior"
  {:name  :blueprint.handler/final
   :leave :response})

(def route
  "Routing interceptor, needs router from `blueprint.router/generate-router`"
  {:name    :blueprint.handler/route
   :spec    ::definition
   :builder (fn [this {:blueprint.core/keys [definition]}]
              (assoc this
                     :enter (interceptor/lens (:router definition)
                                              [:request])))})


(def not-found
  "Quit the chain when the handler is not found."
  {:name :blueprint.handler/not-found
   :enter
   (fn [ctx]
     (let [handler (get-in ctx [:request :handler])]
       (when (identical? handler :blueprint.core/not-found)
         (ex/ex-not-found! "Not found"))
       ctx))})


(def normalize
  "Request normalizer, merges params coming from the body,
   path, or query string and produces a single input map.
   Assoc's the handler name to allow command spec evaluation"
  {:name  :blueprint.handler/normalize
   :enter (-> (fn [{:keys [get-params body-params path-params handler]
                    :as request}]
                (merge (select-keys request [:blueprint.router/options])
                       get-params
                       path-params
                       body-params
                       {:handler handler}))
              (interceptor/lens [:request]))})

(def add-original-request
  "Store original request"
  {:name  :blueprint.handler/original-request
   :enter #(assoc % ::original-request (:request %))})

(def original-request
  "Retrieve original request from a context"
  ::original-request)

(defn ^:no-doc transform-input
  [{:keys [request]} ns]
  (let [handler-id (:handler request)
        spec (core/command-handler-spec ns handler-id)
        coerced (cx/coerce spec request)]
    (when (not (s/valid? spec coerced))
      (log/debugf "Input does not conform: %s"
                  (with-out-str (s/explain spec coerced)))
      (throw (ex/ex-invalid-spec spec coerced)))
    coerced))

(def transform
  "Validates and coerces normalized input as per spec if possible or throws"
  {:name    :blueprint.handler/transform
   :spec    ::definition
   :builder
   (fn [this {:blueprint.core/keys [definition]}]
     (let [ns (:blueprint.registry/ns definition)]
       (assoc this
              :enter
              (fn [ctx]
                (assoc ctx :request (transform-input ctx ns))))))})

(def wrap-response-body
  "Wrap response from handler in a valid ring response map if none is found"
  {:name  :blueprint.handler/wrap-response-body
   :leave (interceptor/lens #(cond->> %
                               (not (and (map? %)
                                         (contains? % :body)))
                               (array-map :body))
                            [:response])})

(defn prepare-payload
  "A helper for `handler` which pulls appropriate keys from the context to
   prepare the request payload.

   By default (for a nil path spec), the value stored at `:request` in the
   context is extracted.

   For a value of `:blueprint.handler/raw`, the context itself is passed down.

   If a collection of keys is given, it is looked up in the context.

   The last possible specification allows fetching keys in various places in
   the context to build the resulting request payload. In this case the
   specification is given as a map of path to path, with the option to
   ask for merging of the result:

       {[:request]                      :blueprint.handler/merge
        [:subsystem :some/generated-id] [:important-id]}

   In this map paths are always expressed as collection of keys.
   In the target specification, a value of `:blueprint.handler/merge`
   will merge the result of the lookup onto the request payload, rather
   than associng it."
  [paths context]
  (cond
    (nil? paths)
    (:request context)

    (= :blueprint.handler/raw paths)
    context

    (sequential? paths)
    (get-in context paths)

    :else
    (reduce-kv (fn [result path target]
                 (let [v (get-in context path)]
                   (if (= :blueprint.handler/merge target)
                     (merge result v)
                     (cond-> result (some? v) (assoc-in target v)))))
               {}
               paths)))

(def handler
  "An interceptor which defers to a provided handler"
  {:name    :blueprint.handler/handler
   :spec    ::handler
   :builder (fn [this {:blueprint.handler/keys [handler request-paths]}]
              (assoc this :enter (interceptor/out
                                  (comp handler
                                        (partial prepare-payload request-paths))
                                  [:response])))})

(defn- log-request
  "Logs the request, including execution for response"
  [{:keys [::logger-ts ::original-request response] :as ctx}]
  (let [elapsed (- (System/currentTimeMillis) logger-ts)
        {:keys [uri request-method]} original-request
        {:keys [status]} response]
    ;; sometimes status is null, I think aleph emits 200 in this case
    (log/infof "%s %s status %s [%sms]" request-method uri status elapsed)
    (dissoc ctx ::logger-ts)))

(def logger
  "A generic request/response/error logger that logs request and elapsed time.
  Rethrows any error"
  {:name  :blueprint.handler/logger
   :enter (fn [ctx]   (assoc ctx ::logger-ts (System/currentTimeMillis)))
   :leave (fn [ctx]   (log-request ctx) ctx)
   :error (fn [ctx e] (log-request ctx) (throw e))})

(s/def ::definition (s/keys :req [:blueprint.core/definition]))
(s/def :blueprint.handler/request-paths
  (s/or :k #{:blueprint.handler/raw}
        :s coll?
        :m (s/map-of coll? (s/or :c coll?
                                 :m #{:blueprint.handler/merge}))))

(s/def ::handler    (s/keys :req [:blueprint.handler/handler]
                            :opt [:blueprint.handler/request-paths]))
