(ns blueprint.handler.error
  (:require [exoscale.ex :as ex]
            [clojure.spec.alpha :as s]
            [clojure.tools.logging :as log]
            [clojure.core.protocols :as p]))

;; Utilities
;; =========

(defn- find-message
  "Finds most appropriate message to expose within exception stack."
  [e]
  (loop [e e]
    (let [msg (ex-message e)]
      (cond
        (some? msg) msg
        (some? e)   (recur (ex-cause e))))))

;; Translation of exceptions to responses
;; ======================================

(defmulti ex->format
  #(some-> % ex-data :type)
  :hierarchy ex/hierarchy)

(defmethod ex->format ::ex/incorrect    [_]
  {:http/status 400 :http/message "bad request"  ::log? false})

(defmethod ex->format ::ex/not-found    [_]
  {:http/status 404 :http/message "not found"    ::log? false})

(defmethod ex->format ::ex/invalid-spec [_]
  {:http/status 400 :http/message "invalid data" ::log? false})

(defmethod ex->format ::ex/forbidden    [_]
  {:http/status 403 :http/message "forbidden"    ::log? false})

(defmethod ex->format ::ex/unsupported  [_]
  {:http/status 405 :http/message "unsupported"  ::log? false})

(defmethod ex->format ::ex/conflict     [_]
  {:http/status 409 :http/message "conflict"     ::log? false})

(defmethod ex->format ::ex/unavailable  [_]
  {:http/status 504 :http/message "server error" ::log? true})

(defmethod ex->format ::ex/busy         [_]
  {:http/status 503 :http/message "busy"         ::log? true})

(defmethod ex->format :default          [_]
  {:http/status 500 :http/message "server error" ::log? true})

(defn ex->backward-compat-format
  [e]
  (when (ex/type? ::ex/user-exposable e)
    {::reveal? true}))

(defn default-logger [ctx e]
  (log/error e "while processing request" (str (:request-id ctx))
             ":" (find-message e)))

(defn make-error-handler
  "Process incoming exception, log if necessary with a provided `logger`
  and produce a valid error message from it.

  `::reveal?` dictates whether we will take the original
  `ex-message` (when true) or another value. `::reveal?` can be
  defined at multiple levels.

  `::datafy?` dictates whether we will convert the exception to data or not.
  Default is `false`

  * takes value of the ex-data
  * or takes the one returned by `ex->format` multimethod
  * or takes the one from the interceptor default (defaults to false)

  When `::reveal?` is false, if the exception ex-data contains an
  `:http/message` at any level this key value will be displayed,
  otherwise it will take the `:http/message` returned by the
  `ex->format` multimethod.
  "
  [{::keys [logger reveal? datafy?]
    :or {logger default-logger
         reveal? false
         datafy? false}}]
  (fn [ctx e]
    (let [{:http/keys [headers status body message]
           ::keys [log? reveal? datafy?]
           :or {reveal? reveal?   ; ex-data could have reveal? otherwise take interceptor's value
                datafy? datafy?}} ; same for datafy
          (merge (ex->format e) ; we could have reveal? at this level
                 (ex->backward-compat-format e) ; till look at ::ex/user-exposable
                 (ex-data e)) ; we could have reveal? at this level too

          ;; either we already have a body; or we want to datafy the error; or create one
          message  (if reveal? (find-message e) message)
          new-body (or body
                     (and datafy? (satisfies? p/Datafiable e) (p/datafy e))
                     {:message message})]

      (when log?
        (logger ctx e))

      (update ctx
              :response
              (fn [response]
                (cond-> (assoc response
                               :status status
                               :body new-body)
                  (some? headers)
                  (assoc-in [:response :headers] headers)))))))

(def interceptor
  {:name    ::error
   :spec    ::logger-fn
   :builder (fn [this opts]
              (assoc this
                     :error
                     (make-error-handler opts)))})

(defn not-found
  "A helper to build a properly formatted not-found exception"
  ([]
   ;; specify both since we need to support reveal? true/false
   (ex/ex-not-found "not found"
                    {:http/message "not found"}))
  ([msg]
   (ex/ex-not-found msg {:http/message msg})))

(defn last-ditch-response
  [e]
  (let [{:http/keys [status body message headers]}
        (try (ex->format e)
             (catch Exception _ #:http{:status 500 :body "server error"}))]
    (cond-> {:status status :body (or message body)}
      (some? headers)
      (assoc :headers headers))))

(def last-ditch
  {:name    ::last-ditch
   :error   (fn [ctx e]
              (log/error e "early/late error during interceptor chain run"
                         (ex-message e))
              (last-ditch-response e))})

(s/def ::logger fn?)
(s/def ::logger-fn (s/keys :opt [::logger]))
