(ns blueprint.handler.format
  "Content-type negotiation and parsing, thanks to muuntaja"
  (:require [exoscale.interceptor :as interceptor]
            [exoscale.ex          :as ex]
            [muuntaja.core        :as m]
            [muuntaja.format.core :as mfc]
            [muuntaja.format.json :as mfj]
            [jsonista.core        :as json]
            [exoscale.cloak]
            [clojure.string :as str])
  (:import (com.fasterxml.jackson.databind MapperFeature
                                           SerializationFeature)))

(defn safe-json-mapper
  [{::keys [allow-namespaced?]}]
  (doto (json/object-mapper
         (if allow-namespaced?
           {:decode-key-fn true}
           {:decode-key-fn #(keyword (if-let [idx (str/last-index-of % "/")]
                                       (subs % (inc idx))
                                       %))}))
    (.configure SerializationFeature/FAIL_ON_EMPTY_BEANS false)
    (.configure MapperFeature/AUTO_DETECT_GETTERS false)
    (.configure MapperFeature/AUTO_DETECT_IS_GETTERS false)
    (.configure MapperFeature/AUTO_DETECT_SETTERS false)
    (.configure MapperFeature/AUTO_DETECT_FIELDS false)
    (.configure MapperFeature/DEFAULT_VIEW_INCLUSION false)))

(defn muuntaja-instance
  [config]
  (m/create
   (assoc-in m/default-options
             [:formats "application/json"]
             (mfc/map->Format
              {:name "application/json"
               :decoder [mfj/decoder {:mapper (safe-json-mapper config)}]
               :encoder [mfj/encoder {:mapper (safe-json-mapper config)}]}))))

(def known-error-types
  "An exhaustive list of error types thrown by the muntaaja library"
  #{:muuntaja/decode
    :muuntaja/request-charset-negotiation
    :muuntaja/response-charset-negotiation
    :muuntaja/response-format-negotiation})

;; Install derivations for muuntaja, by declaring
;; that the above error types belong to the :exoscale.ex/incorrect
;; category, such errors will result in 400 being thrown
(doseq [error-type known-error-types]
  (ex/derive error-type ::ex/incorrect))

(defn- format-enter
  [config]
  (let [m (muuntaja-instance config)]
    (fn [{:keys [request] :as context}]
      (let [request (m/negotiate-and-format-request m
                                                    request)]
        (assoc context ::request request :request request)))))

(defn- format-leave
  [config]
  (let [m (muuntaja-instance config)]
    (fn [{::keys [request] :keys [response]}]
      (m/format-response m request response))))

(def enter
  "An enter interceptor which negotiates content type based on request
  headers and stores that information in the context.
  The `:blueprint.handler.format/allow-namespaced?` option can
  be used in the configuration in order to keep or not the namespace
  during json serialization."
  {:name  :blueprint.handler/format-enter
   :builder (fn [this config]
              (assoc this :enter (format-enter config)))})

(def leave
  "A leave interceptor which uses the information produced in `format-enter`
  and serializes the HTTP response correctly."
  {:name  :blueprint.handler/format-leave
   :builder (fn [this config]
              (assoc this
                     :leave
                     (interceptor/out (format-leave config) [:response])))})
