(ns blueprint.handler.default
  "Implementation of the default interceptors.
   See `blueprint.handler` for reasoning"
  (:require [clojure.spec.alpha   :as s]
            [spec-tools.core      :as st]
            [exoscale.ex          :as ex]
            [exoscale.interceptor :as interceptor]
            blueprint.core))

(def final
  "An interceptor which extracts the response key to conform to ring behavior"
  {:name  :blueprint.handler/final
   :leave :response})



(def route
  "Routing interceptor, needs router from `blueprint.router/generate-router`"
  {:name    :blueprint.handler/route
   :spec    ::definition
   :builder (fn [this {:blueprint.core/keys [definition]}]
              (assoc this :enter (interceptor/lens (:router definition)
                                                   [:request])))})

(def normalize
  "Request normalizer, merges params coming from the body,
   path, or query string and produces a single input map.
   Assoc's the handler name to allow command spec evaluation"
  {:name  :blueprint.handler/normalize
   :enter (-> (fn [{:keys [get-params body-params path-params handler]
                    :as request}]
                (merge (select-keys request [:blueprint.router/options])
                       get-params
                       path-params
                       body-params
                       {:handler handler}))
              (interceptor/lens [:request]))})

(def add-original-request
  "Store original request"
  {:name  :blueprint.handler/original-request
   :enter #(assoc % ::original-request (:request %))})

(def original-request
  "Retrieve original request from a context"
  ::original-request)

(defn- ^:no-doc transform-input
  [spec input]
  (let [decoded (st/decode spec input st/json-transformer)]
    (when (= decoded ::s/invalid)
      (binding [st/*transformer* st/json-transformer
                st/*encode?*     false]
        (throw (ex/ex-invalid-spec spec input))))
    decoded))

(def transform
  "Validates and coerces normalized input as per spec if possible or throws"
  {:name    :blueprint.handler/transform
   :spec    ::definition
   :builder (fn [this {:blueprint.core/keys [definition]}]
              (let [spec (get-in definition [:specs :handler])]
                (assoc this :enter (-> (partial transform-input spec)
                                       (interceptor/lens [:request])))))})

(def wrap-response-body
  "Wrap response from handler in a valid ring response map if none is found"
  {:name  :blueprint.handler/wrap-response-body
   :leave (interceptor/lens #(cond->> %
                               (not (and (map? %) (contains? % :body)))
                               (array-map :body))
                            [:response])})

(def handler
  "An interceptor which defers to a provided handler"
  {:name    :blueprint.handler/handler
   :spec    ::handler
   :builder (fn [this {:blueprint.handler/keys [handler]}]
              (assoc this :enter (-> handler
                                     (interceptor/in [:request])
                                     (interceptor/out [:response]))))})

(s/def ::definition (s/keys :req [:blueprint.core/definition]))
(s/def ::handler    (s/keys :req [:blueprint.handler/handler]))
