(ns blueprint.handler.default
  "Implementation of the default interceptors.
   See `blueprint.handler` for reasoning"
  (:require [blueprint.core :as core]
            [clojure.spec.alpha :as s]
            [clojure.string :as str]
            [clojure.tools.logging :as log]
            [exoscale.coax :as cx]
            [exoscale.ex :as ex]
            [exoscale.interceptor :as interceptor]
            [exoscale.specs :as xs])
  (:import (java.io InputStream ByteArrayOutputStream ByteArrayInputStream)))

;; bytes
(derive :unit/b :unit/B)
(derive :unit/byte :unit/B)
(derive :unit/bytes :unit/B)

;; kilobytes
(derive :unit/kb :unit/KB)
(derive :unit/kilobyte :unit/KB)
(derive :unit/kilobytes :unit/KB)

;; megabytes
(derive :unit/mb :unit/MB)
(derive :unit/megabyte :unit/MB)
(derive :unit/megabytes :unit/MB)

(defmulti payload-size (fn [unit _value] (keyword "unit" (name unit))))
(defmethod payload-size :unit/B [_ value] value)
(defmethod payload-size :unit/KB [_ value] (* 1024 value))
(defmethod payload-size :unit/MB [_ value] (payload-size :KB (* 1024 value)))
(defmethod payload-size :default [_ _]
  (ex/ex-incorrect! "Invalid payload size unit.
                     Valid values are :b, :kb and :mb"))

(defn- clone-input-stream!
  "it will copy input-stream bytes into ByteArrayInputStream but respecting the number max-bytes allowed.
   if input-stream length is greater than 'max-bytes', http bad request exception will be raised"
  [max-bytes ^java.io.InputStream input-stream]
  (let [tmp ^bytes (byte-array 1024)
        output-stream ^java.io.OutputStream (ByteArrayOutputStream.)
        message (format "can't accept body payload bigger than %s bytes" max-bytes)]
    (loop [bytes-read (.read input-stream tmp 0 1024)
           counter bytes-read]
      (when (> counter max-bytes)
        (ex/ex-incorrect! message {:http/message message}))

      (if-not (pos-int? bytes-read)
        (ByteArrayInputStream. (.toByteArray output-stream))
        (do
          (.write output-stream tmp 0 bytes-read)
          (recur (.read input-stream tmp 0 1024)
                 (+ counter bytes-read)))))))

(defn- inputstream->resettable
  "receives on input stream and returns an resettable input stream.
   if the actual input stream is already resettable it is immediately returned
   otherwise we will drain the stream into another resettable input stream.
   "
  [{:keys [unit size]} ^InputStream input-stream]
  (when input-stream
    (if (.markSupported ^InputStream input-stream)
      input-stream
      (clone-input-stream! (payload-size unit size) input-stream))))

(defn resettable-inputstream [payload-opts ctx]
  (update-in ctx [:request :body] (partial inputstream->resettable payload-opts)))

(def wrap-input-stream
  "assert the body request is an input stream instance
   that is resettable, which means supports reset operation."
  {:name :blueprint.handler/wrap-input-stream
   :spec :wrap-input-stream/spec
   :builder (fn [this {:blueprint.handler/keys [wrap-input-stream]}]
              (let [{:keys [enabled? max-payload-size payload-size-unit]
                     :or {enabled? false
                          max-payload-size 1024
                          payload-size-unit :KB}} wrap-input-stream]
                (assoc this :enter (-> (partial inputstream->resettable {:unit payload-size-unit
                                                                         :size max-payload-size})
                                       (interceptor/lens [:request :body])
                                       (interceptor/when (constantly enabled?))))))})

(def final
  "An interceptor which extracts the response key to conform to ring behavior"
  {:name :blueprint.handler/final
   :leave :response})

(def route
  "Routing interceptor, needs router from `blueprint.router/generate-router`"
  {:name :blueprint.handler/route
   :spec ::definition
   :builder (fn [this {:blueprint.core/keys [definition]}]
              (assoc this
                     :enter (interceptor/lens (:router definition)
                                              [:request])))})

(defn get-root-paths
  [prefix]
  (cond
    (str/blank? prefix)
    #{"/"}
    (str/ends-with? prefix "/")
    #{prefix (str/join (butlast prefix))}
    :else
    #{prefix (str prefix "/")}))

(defn- service-desc-fn
  [service-desc-uri definition]
  (let [prefix (get-in definition [:blueprint.options :path-prefix])
        link-paths (get-root-paths prefix)
        desc-path (cond->> service-desc-uri (some? prefix) (str prefix))
        paths (conj link-paths desc-path)
        service-link (format "<%s>; rel=\"service-desc\"" desc-path)]
    (fn [{:keys [request] :as ctx}]
      (let [{:keys [request-method uri handler]} request]
        ;; Should this work without the router?
        (if (and (identical? handler :blueprint.core/not-found)
                 (identical? request-method :get)
                 (contains? paths uri))
          (interceptor/terminate
           (assoc ctx :response
                  (if (contains? link-paths uri)
                    {:status 204
                     :headers {:link service-link}}
                    {:status 200
                     :headers {:Access-Control-Allow-Origin "*"}
                     :body (:openapi definition)})))
          ctx)))))

(def service-desc
  "Service descriptor interceptor, happens after routing on `:not-found` errors"
  {:name :blueprint.handler/service-desc
   :builder (fn [this {:blueprint.openapi/keys [description-path
                                                add-description?]
                       :blueprint.core/keys [definition]
                       :or {description-path "/openapi.json"
                            add-description? true}}]
              (cond-> this
                add-description?
                (assoc :enter (service-desc-fn description-path definition))))})

(def not-found
  "Quit the chain when the handler is not found."
  {:name :blueprint.handler/not-found
   :enter
   (fn [ctx]
     (let [handler (get-in ctx [:request :handler])]
       (when (identical? handler :blueprint.core/not-found)
         (ex/ex-not-found! "Not found"))
       ctx))})

(defn- safe-request-keys
  [handler bp-def f val]
  (xs/select-keys (or val {}) (f bp-def handler)))

(s/def :blueprint.core/merge-result? boolean?)

(def normalize
  "Request normalizer, merges params coming from the body,
   path, or query string and produces a single input map.
   Assoc's the handler name to allow command spec evaluation"
  {:name :blueprint.handler/normalize
   :builder (fn [this {:blueprint.core/keys [definition merge-result?]}]
              (assoc this
                     :enter
                     (-> (fn [request]
                           (let [{:keys [get-params body-params path-params handler]} request
                                 bp-def (:blueprint.registry/ns definition)
                                 get-params-safe  (safe-request-keys handler bp-def
                                                                     core/command-params-spec
                                                                     get-params)
                                 path-params-safe (safe-request-keys handler bp-def
                                                                     core/command-path-spec
                                                                     path-params)
                                 body-params-safe (safe-request-keys handler bp-def
                                                                     core/command-input-spec
                                                                     body-params)

                                 merged-request (-> (if merge-result? request {})
                                                    (merge (select-keys request [:blueprint.router/options])
                                                           get-params-safe
                                                           body-params-safe
                                                           path-params-safe))
                                 spec    (core/command-handler-spec bp-def handler)
                                 coerced (cx/coerce spec merged-request)]

                             (when (not (s/valid? spec coerced))
                               (log/debugf "Input does not conform: %s" (with-out-str (s/explain spec coerced)))
                               (throw (ex/ex-invalid-spec spec coerced)))

                             (-> coerced
                                 (assoc :handler handler
                                        :get-params (select-keys coerced (keys get-params-safe))
                                        :path-params (select-keys coerced (keys path-params-safe))
                                        :body-params (select-keys coerced (keys body-params-safe))))))
                         (interceptor/lens [:request]))))})

(def add-original-request
  "Store original request"
  {:name  :blueprint.handler/original-request
   :enter #(assoc % ::original-request (:request %))})

(def original-request
  "Retrieve original request from a context"
  ::original-request)

(defn ^:no-doc transform-input
  [{:keys [request]} ns]
  (let [handler-id (:handler request)
        spec       (core/command-handler-spec ns handler-id)
        coerced    (cx/coerce spec request)]
    (when (not (s/valid? spec coerced))
      (log/debugf "Input does not conform: %s"
                  (with-out-str (s/explain spec coerced)))
      (throw (ex/ex-invalid-spec spec coerced)))
    ;; need to copy the params to the right place again (same as normalize ix)
    ;; because coercion only works at root level
    ;; we can't just change the ix order to run normalize after because this relies on :handler
    ;; && also clients may rely on existing ordering of ixs
    (-> coerced
        (assoc :get-params  (select-keys coerced (keys (:get-params coerced))))
        (assoc :body-params (select-keys coerced (keys (:body-params coerced))))
        (assoc :path-params (select-keys coerced (keys (:path-params coerced)))))))

(def transform
  "Validates and coerces normalized input as per spec if possible or throws"
  {:name :blueprint.handler/transform
   :spec ::definition
   :builder
   (fn [this {:blueprint.core/keys [definition]}]
     (let [ns (:blueprint.registry/ns definition)]
       (assoc this
              :enter
              (fn [ctx]
                (assoc ctx :request (transform-input ctx ns))))))})

(def wrap-response-body
  "Wrap response from handler in a valid ring response map if none is found"
  {:name :blueprint.handler/wrap-response-body
   :leave (interceptor/lens #(cond->> %
                               (not (and (map? %)
                                         (contains? % :body)))
                               (array-map :body))
                            [:response])})

(defn prepare-payload
  "A helper for `handler` which pulls appropriate keys from the context to
   prepare the request payload.

   By default (for a nil path spec), the value stored at `:request` in the
   context is extracted.

   For a value of `:blueprint.handler/raw`, the context itself is passed down.

   If a collection of keys is given, it is looked up in the context.

   The last possible specification allows fetching keys in various places in
   the context to build the resulting request payload. In this case the
   specification is given as a map of path to path, with the option to
   ask for merging of the result:

       {[:request]                      :blueprint.handler/merge
        [:subsystem :some/generated-id] [:important-id]}

   In this map paths are always expressed as collection of keys.
   In the target specification, a value of `:blueprint.handler/merge`
   will merge the result of the lookup onto the request payload, rather
   than associng it."
  [paths context]
  (cond
    (nil? paths)
    (:request context)

    (= :blueprint.handler/raw paths)
    context

    (sequential? paths)
    (get-in context paths)

    :else
    (reduce-kv (fn [result path target]
                 (let [v (get-in context path)]
                   (if (= :blueprint.handler/merge target)
                     (merge result v)
                     (cond-> result (some? v) (assoc-in target v)))))
               {}
               paths)))

(def handler
  "An interceptor which defers to a provided handler"
  {:name :blueprint.handler/handler
   :spec ::handler
   :builder (fn [this {:blueprint.handler/keys [handler request-paths]}]
              (assoc this :enter (interceptor/out
                                  (comp handler
                                        (partial prepare-payload request-paths))
                                  [:response])))})

(defn- log-request
  "Logs the request, including execution for response"
  [{:keys [:blueprint.handler.logger/level
           :blueprint.handler.logger/ts
           ::original-request
           response]
    :as ctx}]
  (let [elapsed (- (System/currentTimeMillis) ts)
        {:keys [uri request-method]} original-request
        {:keys [status]} response]
    ;; Sometimes status is null, I think aleph emits 200 in this case
    (log/logf level "%s %s status %s [%sms]" request-method uri (or status 200) elapsed)
    (dissoc ctx :blueprint.handler.logger/ts :blueprint.handler.logger/level)))

(s/def :blueprint.handler.logger/level #{:trace :debug :info :warn :error})
(s/def :blueprint.handler.logger/prepare-fn ifn?)
(s/def :blueprint.handler.logger/log-fn ifn?)
(s/def ::logger (s/keys :opt [:blueprint.handler.logger/level
                              :blueprint.handler.logger/prepare-fn
                              :blueprint.handler.logger/log-fn]))

(defn- build-logger
  [this {:blueprint.handler.logger/keys [level prepare-fn log-fn]
         :or {level :info
              prepare-fn (constantly {})
              log-fn log-request}}]
  (assoc this
         :enter (fn [ctx] (assoc ctx
                                 :blueprint.handler.logger/context (prepare-fn (::original-request ctx))
                                 :blueprint.handler.logger/level level
                                 :blueprint.handler.logger/ts (System/currentTimeMillis)))
         :leave (interceptor/discard log-fn)
         :error (fn [ctx e] (log-fn ctx) (throw e))))

(def logger
  "A generic request/response/error logger that logs request and elapsed time.
  Rethrows any error"
  {:name :blueprint.handler/logger
   :spec ::logger
   :builder build-logger})

(s/def ::definition (s/keys :req [:blueprint.core/definition]))
(s/def :blueprint.handler/request-paths
  (s/or :k #{:blueprint.handler/raw}
        :s coll?
        :m (s/map-of coll? (s/or :c coll?
                                 :m #{:blueprint.handler/merge}))))

(s/def ::handler (s/keys :req [:blueprint.handler/handler]
                         :opt [:blueprint.handler/request-paths]))

(s/def :wrap-input-stream/max-payload-size pos-int?)
(s/def :wrap-input-stream/payload-size-unit #{:B :KB :MB
                                              :b :kb :mb
                                              :byte :kilobyte :megabyte
                                              :bytes :kilobytes :megabytes})
(s/def :wrap-input-stream/enabled? boolean?)
(s/def :wrap-input-stream/spec (s/keys :opt-un [:wrap-input-stream/enabled?
                                                :wrap-input-stream/max-payload-size
                                                :wrap-input-stream/payload-size-unit]))
