(ns blueprint.openapi
  (:require [jsonista.core          :as json]
            [camel-snake-kebab.core :as csk]))

(defn build-ref
  [resource]
  {:$ref (format "#/components/schemas/%s" (csk/->PascalCaseString resource))})

(defmulti build-pred  :pred)
(defmethod build-pred 'any?     [_] {:type "object" :additional-properties true})
(defmethod build-pred 'uuid?    [_] {:type "string" :format "uuid"} )
(defmethod build-pred 'string?  [_] {:type "string"})
(defmethod build-pred 'boolean? [_] {:type "boolean"})
(defmethod build-pred 'inst?    [_] {:type "string" :format "date-time"})
(defmethod build-pred 'pos-int? [_] {:type "integer" :format "int64"})

(defmulti build-schema :type)

(defmethod build-schema :pred
  [input]
  (build-pred input))

(defmethod build-schema :>
  [{:keys [val]}]
  {:type "integer"
   :format "int64"
   :minimum val
   :exclusiveMinimum true})

(defmethod build-schema :>=
  [{:keys [val]}]
  {:type "integer"
   :format "int64"
   :minimum val
   :exclusiveMinimum false})

(defmethod build-schema :<=
  [{:keys [val]}]
  {:type "integer"
   :format "int64"
   :maximum val
   :exclusiveMaximum val})

(defmethod build-schema :<
  [{:keys [val]}]
  {:type "integer"
   :format "int64"
   :maximum val
   :exclusiveMaximum true})

(defmethod build-schema :not=
  [{:keys [val]}]
  {:not
   {:type "integer"
    :format "int64"
    :minimum val
    :maximum val
    :exclusiveMinimum true
    :exclusiveMaximum true}})

(defmethod build-schema :enum
  [{:keys [values]}]
  {:type "string" :enum (map name values)})

(defmethod build-schema :reference
  [{:keys [reference]}]
  (build-ref reference))

(defmethod build-schema :map-of
  [{:keys [val-def]}]
  {:type "object" :additional-properties (build-schema val-def)})

(defmethod build-schema :coll-of
  [{:keys [def]}]
  {:type "array" :items (build-schema def)})

(defmethod build-schema :and
  [{:keys [defs]}]
  {:all-of (mapv build-schema defs)})

(defmethod build-schema :multi
  [{:keys [alternatives]}]
  {:any-of
   (mapv build-schema (map :def alternatives))})

(defn build-attr
  [{:keys [attribute def opts]}]
  (cond-> (build-schema def)
    (:ro? opts) (assoc :readOnly true)))

(defmethod build-schema :map
  [{:keys [attributes]}]
  {:type       "object"
   :properties (reduce #(assoc %1 (:attribute %2) (build-attr %2)) {} attributes)})

(defmethod build-schema :default
  [input]
  {:type "unknown" :input input})


(defn build-arg
  [{:keys [name def]}]
  {:in       "path"
   :required true
   :name     (clojure.core/name name)
   :schema   (build-schema def)})

(defn build-param
  [[name def]]
  {:in       "query"
   :required false
   :name     (clojure.core/name name)
   :schema   (build-schema def)})

(defn build-endpoint
  [elems]
  (->>
   (for [[t e] elems]
     (if (= :string t)
       e
       (format "{%s}" (name (:name e)))))
   (reduce str)))

(defn build-output
  [code def]
  {:description (str code)
   :content     {"application/json" {:schema (build-schema def)}}})

(defn filter-args
  [elems]
  (for [[t e] elems :when (= :arg t)] e))

(defn generate-path
  [{:keys [op tags input output params path desc] :as cmd}]
  (let  [{:keys [method elems]} path
         args                   (filter-args elems)
         body?                  (some? input)
         endpoint               (build-endpoint elems)]
    (cond->
        {:method       (name method)
         :path         endpoint
         :tags         (mapv name tags)
         :responses    (reduce-kv #(assoc %1 %2 (build-output %2 %3)) {} output)
         :operation-id op
         :description  (:desc cmd)
         :parameters   (concat (mapv build-arg args)
                               (mapv build-param params))
         :summary      (or (:summary cmd) "")}
      body?
      (assoc :requestBody
             {:required true
              :content  {"application/json" {:schema (build-schema input)}}}))))

(defn generate-schemas
  [resources]
  (reduce-kv #(assoc %1 (csk/->PascalCaseString %2) (build-schema %3))
             {}
             resources))

(defn assoc-path
  [m op input]
  (let [{:keys [path method] :as cmd} (generate-path input)]
    (assoc-in m [path method]
              (-> cmd
                  (dissoc :path :method)
                  (assoc :operation-id (name op))))))

(defn generate-tags
  [tags]
  (reduce-kv #(conj %1 (assoc %3 :name (name %2))) [] tags))

(defn generate-openapi
  [{:keys [servers info tags resources commands] :as api-def}]
  (assoc api-def :openapi
         {:openapi    "3.0.0"
          :info       info
          :tags       (generate-tags tags)
          :components {:schemas (generate-schemas resources)}
          :servers    servers
          :paths      (reduce-kv assoc-path {} commands)}))

(defn json-serialize
  [{:keys [openapi]}]
  (json/write-value-as-string
   openapi
   (json/object-mapper {:encode-key-fn csk/->camelCaseString})))

(defn write-to-file
  [api-def path]
  (spit path (json-serialize api-def)))

(comment

  (require 'blueprint.core)

  (def my-api
    (-> (blueprint.core/parse api-description)
        (generate-openapi)))

  (write-to-file my-api "/home/pyr/foo.json")
  my-api










  )
