(ns blueprint.openapi
  "Generates the OAS spec from the blueprint definition.
  Reference docs: https://swagger.io/specification/

  OAS keys are already camelCased."
  (:require [blueprint.registry :as reg]
            [clojure.set :as set]
            [clojure.walk :as walk]
            [jsonista.core :as json])
  (:import [clojure.lang Named]))

(defn build-ref
  [resource]
  {:$ref (format "#/components/schemas/%s" (name resource))})

(defmulti build-pred  :pred)
(defmethod build-pred 'any?     [_] {:type "object" :additionalProperties true})
(defmethod build-pred 'uuid?    [_] {:type "string" :format "uuid"})
(defmethod build-pred 'string?  [_] {:type "string"})
(defmethod build-pred 'boolean? [_] {:type "boolean"})
(defmethod build-pred 'inst?    [_] {:type "string" :format "date-time"})
(defmethod build-pred 'pos-int? [_] {:type "integer" :format "int64"})
(defmethod build-pred 'number? [_] {:type "number"})

(defmulti build-schema :type)

(defmethod build-schema :pred
  [input]
  (build-pred input))

(defmethod build-schema :>
  [{:keys [floor ceiling]}]
  (if (some? ceiling)
    {:type "integer"
     :format "int64"
     :minimum ceiling
     :maximum floor
     :exclusiveMinimum true
     :exclusiveMaximum true}
    {:type "integer"
     :format "int64"
     :minimum floor
     :exclusiveMinimum true}))

(defmethod build-schema :>=
  [{:keys [floor ceiling]}]
  (if (some? ceiling)
    {:type "integer"
     :format "int64"
     :minimum ceiling
     :maximum floor
     :exclusiveMinimum false
     :exclusiveMaximum false}
    {:type "integer"
     :format "int64"
     :minimum floor
     :exclusiveMinimum false}))

(defmethod build-schema :<=
  [{:keys [floor ceiling]}]
  (if (some? ceiling)
    {:type "integer"
     :format "int64"
     :minimum floor
     :maximum ceiling
     :exclusiveMinimum false
     :exclusiveMaximum false}
    {:type "integer"
     :format "int64"
     :maximum floor
     :exclusiveMaximum false}))

(defmethod build-schema :<
  [{:keys [floor ceiling]}]
  (if (some? ceiling)
    {:type "integer"
     :format "int64"
     :minimum floor
     :maximum ceiling
     :exclusiveMinimum true
     :exclusiveMaximum true}
    {:type "integer"
     :format "int64"
     :maximum floor
     :exclusiveMaximum true}))

(defmethod build-schema :not=
  [{:keys [val]}]
  {:not
   {:type "integer"
    :format "int64"
    :minimum val
    :maximum val
    :exclusiveMinimum true
    :exclusiveMaximum true}})

(defmethod build-schema :enum
  [{:keys [values]}]
  (if (empty? values)
    {:type "string" :enum (map name values)}
    ;;else
    ;; https://swagger.io/docs/specification/data-models/data-types/
    (cond (or (every? #(instance? Named %) values)
              (every? #(instance? CharSequence %) values))
          {:type "string" :enum (doall (map name values))}

          (every? #(instance? Number %) values)
          {:type "number" :enum (into [] values)}

          :else
          {:type "object" :enum (into [] values)})))

(defmethod build-schema :ip
  [_]
  {:type "string"
   :format "ip"})

(defmethod build-schema :ipv4
  [_]
  {:type "string"
   :format "ipv4"})

(defmethod build-schema :ipv6
  [_]
  {:type "string"
   :format "ipv6"})

(defmethod build-schema :reference
  [{:keys [reference]}]
  (build-ref reference))

(defmethod build-schema :map-of
  [{:keys [val-def]}]
  {:type "object" :additionalProperties (build-schema val-def)})

(defmethod build-schema :coll-of
  [{:keys [def]}]
  {:type "array" :items (build-schema def)})

(defmethod build-schema :and
  [{:keys [defs]}]
  {:allOf (mapv build-schema defs)})

(defmethod build-schema :multi
  [{:keys [alternatives]}]
  {:anyOf
   (mapv build-schema (map :def alternatives))})

(defn build-attr
  [{:keys [def opts]}]
  (cond-> (build-schema def)
    (:ro? opts) (assoc :readOnly true)))

(defmethod build-schema :map
  [{:keys [attributes]}]
  {:type       "object"
   :properties (reduce #(assoc %1 (:attribute %2) (build-attr %2)) {} attributes)})

(defmethod build-schema :default
  [input]
  {:type "unknown" :input input})

(defn build-arg
  [{:keys [name def]}]
  {:in       "path"
   :required true
   :name     (clojure.core/name name)
   :schema   (build-schema def)})

(defn build-param
  [[name def]]
  {:in       "query"
   :required false
   :name     (clojure.core/name name)
   :schema   (build-schema def)})

(defn build-endpoint
  [elems]
  (->>
   (for [[t e] elems]
     (if (= :string t)
       e
       (format "{%s}" (name (:name e)))))
   (reduce str)))

(defn- prepare-extensions
  "Prefixes all the keys in the `extensions` map with `x-`"
  [extensions]
  (reduce-kv (fn [m k v]
               (assoc m (keyword (str "x-" (name k)))
                      v))
             {}
             ;;stringify will prevent :keyword to camelCase transformation
             ;;when spitting the json
             (walk/stringify-keys extensions)))

(defn- prefix-extensions
  "Extracts `ext-key` extensions out of `m` and prepends
  the openapi `x-` prefix to each extension"
  [m ext-key]
  (let [extensions (get m ext-key)]
    (merge (dissoc m ext-key)
           (prepare-extensions extensions))))

(defn build-output
  [code def output-extensions]
  (let [code-extensions (get output-extensions code)
        prefixed-exts   (prepare-extensions code-extensions)]
    (-> {:description (str code)
         :content     {"application/json" {:schema (build-schema def)}}}
        (merge prefixed-exts))))

(defn filter-args
  [elems]
  (for [[t e] elems :when (= :arg t)] e))

(defn generate-path
  [{:keys [op tags input output params path output-extensions] :as cmd}]
  (let  [{:keys [method elems]} path
         args                   (filter-args elems)
         body?                  (some? input)
         endpoint               (build-endpoint elems)]
    (cond->
     {:method       (name method)
      :path         endpoint
      :tags         (mapv name tags)
      :responses    (reduce-kv #(assoc %1 %2
                                       (build-output %2 %3 output-extensions))
                               {} output)
      :operationId  op
      :description  (:desc cmd)
      :parameters   (concat (mapv build-arg args)
                            (mapv build-param params))
      :summary      (or (:summary cmd) "")}
      body?
      (assoc :requestBody
             {:required true
              :content  {"application/json" {:schema (build-schema input)}}}))))

(defn generate-schemas
  [resources]
  (reduce-kv #(assoc %1 (name %2) (build-schema %3))
             {}
             resources))

(defn assoc-path
  [m op input]
  (let [{:keys [path method] :as cmd} (generate-path input)
        defined-extensions (:extensions input)]
    (assoc-in m [path method]
              (-> cmd
                  (dissoc :path :method)
                  (assoc :operationId (name op))
                  (merge (when defined-extensions
                           (prepare-extensions defined-extensions)))))))

(defn generate-tags
  [tags]
  (->> tags
       (reduce-kv #(assoc %1 %2 (set/rename-keys %3 {:external-docs :externalDocs})) {})
       (reduce-kv #(conj %1 (assoc %3 :name (name %2))) [])
       (map #(prefix-extensions % :extensions))))

(defn- visible? [[_ {:keys [options]}]]
  (get-in options [:openapi?] true))

(defn generate-openapi
  [{:keys      [servers info tags extensions]
    ::reg/keys [resources commands]
    :as        api-def}]
  (assoc api-def
         :has-openapi?
         true
         :openapi
         (-> {:openapi    "3.0.0"
              :info       (prefix-extensions info :extensions)
              :tags       (generate-tags tags)
              :components {:schemas (generate-schemas resources)}
              :servers    servers
              :extensions extensions
              :paths      (reduce-kv assoc-path {} (into {} (filter visible? commands)))}
             (prefix-extensions :extensions))))

(defn- encode-key-fn
  [k]
  (let [n (name k)]
    ;;dont apply camelCase, to ensure
    n))

(def object-mapper
  (json/object-mapper {:encode-key-fn encode-key-fn
                       :pretty true}))

(defn json-serialize
  [{:keys [openapi]}]
  (json/write-value-as-string openapi object-mapper))

(defn write-to-file
  [api-def path]
  (spit path (json-serialize api-def)))

(comment

  (require 'blueprint.core)

  (def my-api
    (-> (blueprint.core/parse api-description)
        (generate-openapi)))

  (write-to-file my-api "/home/pyr/foo.json")
  my-api)
