(ns blueprint.handler.reporter
  "A replacement for the default error logger which depends on a
   configured reporter.

   The intended use for the exposed function is to find its way in a
   server configuration, in the `blueprint.handler.error/logger` key.

   If a reporter is configured - say, through exoscale/mania - you
   can merge:

      {:blueprint.handler.error/logger blueprint.handler.reporter/report}

   onto your server options. If you wish to extract more standard keys
   from your context when reporting to sentry, build the appropriate
   plucking with `report-fn`."
  (:require [spootnik.reporter       :as r]
            [unilog.context          :as ctx]
            [blueprint.handler.error :as error]
            [raven.client            :as raven]
            [clojure.tools.logging   :as log]
            [clojure.spec.alpha      :as s]
            [exoscale.ex             :as ex]))

;; This will allow failing early if the config provided to
;; `pluck-fn` is wrong
(s/def ::paths (s/or :map  (s/map-of keyword? coll?)
                     :coll (s/coll-of keyword?)))

(defn pluck-fn
  "A closure which yields a function to select keys out of the context.

   The path specification may either be a collection, in which case, it must
   contain keywords which will be selected out of the context.

   If paths is a map it is expected to a dictionary of key to path in the
   input."
  [paths]
  (ex/assert-spec-valid ::paths paths)
  (if (map? paths)
    (fn [ctx] (reduce-kv #(assoc %1 %2 (get-in ctx %3)) {} paths))
    #(select-keys % paths)))

(defn report-fn
  "Build a reporting function, valid for use as an alternative logger
   for a blueprint error interceptor. Uses a path specification for
   keys to pluck out of the context

   The path specification may either be a collection, in which case, it must
   contain keywords which will be selected out of the context.

   If paths is a map it is expected to a dictionary of key to path in the
   input."
  [paths]
  (let [pluck (pluck-fn paths)]
    (fn [ctx e]
      (try
        (let [plucked (pluck ctx paths)]
          (ctx/with-context plucked
            (error/default-logger ctx e)
            (when-not (:reporter/prevent? (ex-data e))
              (r/capture! (-> {:data    (ex-data e)
                               :message (ex-message e)}
                              (raven/add-extra! plucked)
                              (raven/add-exception! e))))))
        (catch Exception e
          (log/error e "could not report to sentry"))))))

(def report
  "Default reporting function, only plucks `:request-id` from
   a blueprint context, use `report-fn` to build your reporting
   function if you want more keys to be selected out"
  (report-fn [:request-id]))
