(ns blueprint.router
  (:require [reitit.core        :as r]
            [blueprint.registry :as reg]
            [exoscale.ex        :as ex]))

(defn gen-path
  [elems]
  (reduce
   str
   (for [[_ v] elems]
     (if (string? v) v (format "{%s}" (name (:name v)))))))

(defn prepare-route
  [options routers opname {:keys [path]}]
  (let [path-prefix (get options :path-prefix "")
        {:keys [method elems]} path]
    (update routers method conj [(str path-prefix (gen-path elems)) opname])))

(defn build-router
  [api-def routers]
  (fn [{:keys [request-method uri] :as request}]
    (let [{:keys [data path-params]} (r/match-by-path (get routers request-method)
                                                      uri)]
      (cond-> request
        (some? data)
        (assoc :handler (:name data)
               ::options (get-in api-def [:commands (:name data) :options] {}))

        (nil? data)
        (assoc :handler :blueprint.core/not-found)

        (some? path-params)
        (assoc :path-params path-params)))))

(defn generate-router
  "Takes an api definition that has been parsed by `blueprint.core/parse`"
  ([api-def]
   (generate-router api-def {}))
  ([{::reg/keys [commands] :as api-def} options]
   (assoc api-def
          :router-generated?
          true

          :router
          (build-router
           api-def
           (->> (reduce-kv (partial prepare-route (:blueprint.options api-def))
                           {}
                           commands)
                (reduce-kv #(assoc %1 %2
                                   (r/router %3 (merge {:syntax #{:bracket}}
                                                       options)))
                           {}))))))

(defn build-url
  [elems params]
  (for [e elems]
    (if (string? e) e (str (get params (first e))))))

(defn uri-for
  [api-def nick params]
  (if-let [{[method & elems] :path} (get (:commands api-def) nick)]
    (let [path-prefix (get-in api-def [:blueprint.options :path-prefix])
          url         (reduce str path-prefix (build-url elems params))]
      {:request-method method
       :url            url})
    (throw (ex-info "unknown command" {:type ::ex/not-found}))))
