blueprint: data definitions for APIs
====================================

```clojure
   [exoscale/blueprint-core    "0.1.0"]
   [exoscale/blueprint-openapi "0.1.0"]
   [exoscale/blueprint-router  "0.1.0"]
   [exoscale/blueprint-handler "0.1.0"]
```

Blueprint is a data schema for APIs, with generation support for:

- OpenAPI specifications
- Clojure specs
- Request routers
- Ring handlers

Blueprint relies on the notion of *resources* and *commands*.
Resources are input and output descriptions while commands describe
API endpoints.

At the core of blueprint is a *spec* generator, inspired by
[malli](https://github.com/metosin/malli) but retaining compatibility
with `clojure.spec`.

The full API description structure is as follows:

```clojure
{:resources {:resource-name <resource-description>}
 :commands  {:command-name <command-description>}
 :info      <general-info>
 :tags      <tag-info>
 :servers   <server-info>}
```

With a valid description - as described below - specs, routers,
OpenAPI specifications, and ring handlers can be generated:

```clojure
(def api-description
  '{:resources
    {:job
     [:map
      [:id          {:desc "Job ID" :ro? true}  uuid?]
      [:title       {:desc "Job title" :req? true} string?]
      [:company     {:desc "Company" :req? true} string?]
      [:tags        {:desc "Tags"}                 [:coll-of string?]]
      [:description {:desc "Description"}          string?]]}

    :commands
    {:list-jobs          {:path   [:get "/job"]
                          :desc   "List jobs"
                          :tags   [:job]
                          :params {:company string?}
                          :output {200 [:map [:jobs [:coll-of :job]]]}}
     :show-job-details   {:path   [:get "/job/" [:id uuid?]]
                          :desc   "Show job details"
                          :tags   [:job]
                          :output {200 :job}}
     :remove-job-posting {:path   [:delete "/job/" [:id uuid?]]
                          :desc   "Remove job posting"
                          :tags   [:job]
                          :output {204 any?}}
     :add-job-posting    {:path   [:post "/job"]
                          :desc   "Add job posting"
                          :tags   [:job]
                          :output {200 :job}}}

    :tags
    {:job {:description   "Jobs"
           :external-docs {:url         "/somewhere"
                           :description "more..."}}}

    :servers
    [{:url "https://localhost:8080"}]

    :info
    {:title       "A jobs API"
     :version     "1.0"
     :description "Demo API"}})

(def api-def
  (-> (blueprint.core/parse api-description)
      (blueprint.openapi/generate-openapi)
	  (blueprint.router/generate-router)))
```

### Validating input

Based on the parsed description, specs can be evaluated. Currently
four types of specs are generated:

- Resource specs to validate resources
- Command specs to validate individual commands
- A multi-spec which dispatches on a `:handler` key in the input and
  validates against the appropriate command spec
- A response spec which validates command output

Specs can be looked up with the functions `blueprint.core/resource-spec`,
`blueprint.core/command-spec`, `blueprint.core/handler-spec`, and
`blueprint.core/response-spec`.

### Generating OpenAPI specification

```clojure
(blueprint.openapi/write-to-file api-def "my-api.json")
```

You can then run swagger UI to explore the produced API specification:

```
docker run -p 8080:8080 -e SWAGGER_JSON=/my-api.json -v `pwd`/my-api.json:/my-api.json swaggerapi/swagger-ui
```

### Building a ring handler

Let's assume a basic handler for our job database

```clojure
(def jobs (atom {}))
(defn new-id [] (java.util.UUID/randomUUID))

(defmulti job-handler :handler)

(defmethod job-handler :list-jobs [_] {:jobs (vals @jobs)})
(defmethod job-handler :show-job-details [{:keys [id]}] (get @jobs id))
(defmethod job-handler :remove-job-posting [{:keys [id]}] (swap! jobs dissoc id))
(defmethod job-handler :add-job-posting [job] (let [id (new-id)] (swap! jobs assoc id (assoc job :id id))))
```

We can now expose the above API with a produced handler:

```clojure
(require '[aleph.http :as http])
(require '[blueprint.handler :as handler])
(require '[blueprint.core :as blueprint])
(require '[blueprint.router :as router])

(def srv (-> api-description
             blueprint/parse
			 router/generate-router
			 (handler/generate-ring-handler job-handler)
			 (http/start-server {:port 8080})))
```

The `generate-ring-handler` also accepts an option map as a third parameter. The available options are:

- `default-error-msg`: A default error message for the error handler.
- `interceptor-chain`: A custom interceptor chain for the handler.
- `user-interceptors`: A map containing custom interceptors. These interceptors can be referenced in the `interceptor-chain` variable to be used.

## Describing resources

TBD

## Describing commands

TBD
