# checkmate

A library to handle failure/retries/rate-limiting.

``` clj
(require '[exoscale.checkmate.sync :as cm])
(require '[exoscale.checkmate.condition :as cc])

;; runs at most 3 times, with infinite seq of delays of 1s
(cm/run #(something-brittle)
        [(cc/delayed-retries (repeat 1000)) (cc/max-retries 3)])

;; same but limiting the delays count via delayed-retries
(cm/run #(something-brittle)
        [(cc/delayed-retries (repeat 3 1000))])

;; just run at most 3 times, no delays
(cm/run #(something-brittle)
        [(cc/delayed-retries (repeat 3 1000)) (cc/max-retries 3)])

;; generating delays is easy, they are just seqs:
;; snippet from the condition ns:

(defn constant-backoff-delays [ms]
  (repeat ms))

(defn exponential-backoff-delays [x]
  (iterate #(* Math/E %) x))

(defn progressive-backoff-delays [x]
  (lazy-cat
   (repeat x 100)
   (repeat x 500)
   (repeat x 1500)
   (repeat x 15000)
   (repeat 60000)))

[...]

;;; Rate limiter condition
;; just run at most 3 times, use shared rate-limiter with initial pool size of
;; 10 and allowing only up to 1 retry run per sec after pool exhaustion
(def rl (cc/rate-limiter {:max-errors 10 :error-rate 1000}))

;; these 3 calls share the rate limiter

(cm/run #(something-brittle)
        [rl (cc/max-retries 3)])

(cm/run #(something-brittle-too)
        [rl (cc/max-retries 10)])

(cm/run #(something-brittle-too)
        [rl (cc/max-retries 5)])


;; hooks (error, failure, success)
(cm/run #(something-brittle-too)
        [...]
        {:exoscale.checkmate.hook/error (fn [err] (log/error "boom, retrying" err))   ... })

;; manifold, returns a deferred
(require '[exoscale.checkmate.manifold :as mm])
(mm/run #(something-returning-a-deferred) [...])

;; core.async, returns a promise-chan
(require '[exoscale.checkmate.core-async :as ca])
(ca/run #(something-returning-a-chan) [...])

```

checkmate is made of simple building blocks:

* a runner
* conditions
* condition effects

Conditions control if we should retry and how to setup the state at
first call, and how to update the state upon failures (inc retry
counter for instance). Conditions are in fact, **composable** and
should remain pure (state in/out).

Effects can apply side effects at various stages (ex sleep, update
state machine), they are purely for io, as opposed to Conditions.

The simplest condition would be for max-retries:

``` clj
(defn max-retries
  [max]
  (reify Condition
    (setup! [this state]
      (assoc state :exoscale.checkmate/retries 0))
    (retry? [this state]
      (< (:exoscale.checkmate/retries state) max))
    (update! [this state]
      (update state :exoscale.checkmate/retries inc))))
```

setups state counter per `run` to 0, a check to know if we need to
retry from the state and an `update!` to refresh state after an error


A slightly more involved condition with an effect (simplified version):

``` clj
(defn delayed-retries
  [delays]
  (reify
    Condition
    (setup! [this state]
      (assoc state
             :exoscale.checkmate/delays delays))
    (retry? [this state]
      (seq (:exoscale.checkmate/delays state)))
    (update! [this state]
      (update state :exoscale.checkmate/delays rest))
    ErrorEffect
    (error-effect! [this state]
       (Thread/sleep (-> state :exoscale.checkmate/delays first)))))
```

Effects do not affect state, but they can affect execution, for
instance apply a pause or cause an execution rejection in the case of
the rate limiter. They should not be used for logging as they are tied
to a condition's implementation, hooks are better for this since they
are per call.

## Documentation

[![cljdoc badge](https://cljdoc.xyz/badge/exoscale/checkmate)](https://cljdoc.xyz/d/exoscale/checkmate/CURRENT)

## Installation

checkmate is [available on Clojars](https://clojars.org/exoscale/checkmate).

Add this to your dependencies:

[![Clojars Project](https://img.shields.io/clojars/v/exoscale/checkmate.svg)](https://clojars.org/exoscale/checkmate)

Please check the
[Changelog](https://github.com/mpenet/checkmate/blob/master/CHANGELOG.md)
if you are upgrading.

## License

Copyright © 2020 [Exoscale](https://exoscale.com)
