(ns exoscale.checkmate.condition
  (:require [clojure.core.async :as async]
            [manifold.time :as mt]))

;;; Conditions manipulate/query state

(defprotocol Condition
  (setup! [this state]
    "Setups initial state before first run")
  (retry? [this state]
    "Returns true if we should retry upon error on this run")
  (update! [this state]
    "Update state after error"))

;;; Effects

;; Effects potentially run IO at various stage, they take the
;; condition + state, state after setup will contain ::result of the
;; current call as well
(defprotocol SetupEffect
  (setup-effect! [this state]
    "Effect triggered before Run"))

(defprotocol SuccessEffect
  (success-effect! [this state]
    "Effect triggered upon success"))

(defprotocol ErrorEffect
  (error-effect! [this state]
    "Effect triggered upon error"))

(defprotocol FailureEffect
  (failure-effect! [this state]
    "Effect triggered upon terminal failure"))

;;; Conditions implementations

(defn max-retries
  [max]
  (reify Condition
    (setup! [this state]
      (assoc state :exoscale.checkmate/retries 0))
    (retry? [this state]
      (< (:exoscale.checkmate/retries state) max))
    (update! [this state]
      (update state :exoscale.checkmate/retries inc))))

(defn retry-on
  "Allow retries on (pred ret) -> true, useful to limit retries to a
  class of errors
  ex: (retry-on #(instance? TimeoutException %))"
  [pred]
  (reify Condition
    (setup! [this state] state)
    (retry? [this state] (pred this state))
    (update! [this state] state)))

(defn delayed-retries
  [delays]
  (reify
    Condition
    (setup! [this state]
      (assoc state
             :exoscale.checkmate/delays delays))
    (retry? [this state]
      (seq (:exoscale.checkmate/delays state)))
    (update! [this state]
      (update state :exoscale.checkmate/delays rest))
    ErrorEffect
    (error-effect! [this state]
      (when-let [delay (some-> state :exoscale.checkmate/delays first)]
        ;; we could open this, but meh, with the async proto we were
        ;; talking about maybe
        (case (:exoscale.checkmate/runner state)
          :sync (Thread/sleep delay)
          :core-async (async/timeout delay)
          :manifold (mt/in delay (constantly ::noop)))))))

;;; following are to be used as aruments to delayed-retries
(defn constant-backoff-delays [ms]
  (repeat ms))

(defn exponential-backoff-delays [x]
  (iterate #(* Math/E %) x))

(defn progressive-backoff-delays [x]
  (lazy-cat
   (repeat x 100)
   (repeat x 500)
   (repeat x 1500)
   (repeat x 15000)
   (repeat 60000)))

(defn rate-limited-retries
  "!! This condition should be shared for all runner calls, since it's
  stateful

  Only allows a max rate of errors until it plain rejects calls until
  rate can be satisfied again:

  Example:

  max-errors : 10
  error-rate : 1000ms

  Allows up to 10 successive errors, upon token exhaustion will only
  allow 1 error per second in burst, until it fills again to 10.

  So this allows burst'iness up to a point.

  If you create a lot or rate-limited-retries conditions you need to
  remember to async/close! the associated chan, or pass it as
  argument. The chan is also available via Condition state, in case
  you want to enable temporary burst'iness, early flush or closing"
  [{:keys [max-errors error-rate exception]
    :or {max-errors 10
         error-rate 1000
         exception (fn [condition state]
                     (ex-info "Rejected execution, max rate reached"
                              {:type :exoscale.ex/busy
                               :exoscale.checkmate/condition condition :exoscale.checkmate/state state}))}}]

  (let [ch (async/chan max-errors)
        reject? (atom false)]
    ;; we will create a token emitting chan at desired
    ;; rate, upon errors call we try to poll! from that
    ;; chan, if it fails we know rate is exceeded and we
    ;; can just reject.
    (dotimes [_ max-errors]
      (async/>!! ch :exoscale.checkmate/token))

    (async/go
      ;; then feed at rate
      (loop []
        (async/<! (async/timeout error-rate))
        (when (async/>! ch :exoscale.checkmate/token)
          (reset! reject? false)
          (recur))))

    (reify Condition
      (setup! [this state]
        ;; to allow the user to potentially close/feed it
        (assoc state :exoscale.checkmate/ch ch))
      (update! [this state] state)
      (retry? [this state]
        ;; no token available, no retry
        (async/poll! ch))
      SetupEffect
      (setup-effect! [this state]
        ;; do not allow calls while it's rate limited
        (when @reject?
          (throw (exception this state))))

      FailureEffect
      (failure-effect! [this state]
        ;; stop allowing request until next token tick
        (reset! reject? true)
        (throw (exception this state))))))
