(ns exoscale.checkmate.impl
  (:require [clojure.tools.logging :as log]
            [clojure.spec.alpha :as s]))

(def default-options
  {:exoscale.checkmate.hook/failure
   #(log/errorf "Run failure, aborting: run-id=%s, error=%s, abort-cause=%s, runner=%s"
                (:exoscale.checkmate/run-id %)
                (:exoscale.checkmate/error %)
                (-> % :exoscale.checkmate/abort-cause :exoscale.checkmate/id)
                (:exoscale.checkmate/runner %))
   :exoscale.checkmate.hook/error
   #(log/errorf "Run error, retrying: run-id=%s, error=%s, runner=%s"
                (:exoscale.checkmate/run-id %)
                (:exoscale.checkmate/error %)
                (:exoscale.checkmate/runner %))
   :exoscale.checkmate.hook/success identity})

(defn error-ctx
  [ctx err]
  (assoc ctx :exoscale.checkmate/error err))

(defn success-ctx
  [ctx x]
  (assoc ctx :exoscale.checkmate/success x))

(defn assert-conditions-valid! [conditions]
  (when (s/check-asserts?)
    (run! #(s/assert :exoscale.checkmate/condition %)
          conditions)))

(defn setup-conditions
  [conditions ctx runner]
  (reduce (fn [ctx {:as _cd :exoscale.checkmate/keys [setup]}]
            (cond-> ctx
              (ifn? setup)
              setup))
          (assoc ctx
                 :exoscale.checkmate/runner runner
                 :exoscale.checkmate/run-id (java.util.UUID/randomUUID))
          conditions))

(defn update-conditions
  [conditions ctx]
  (reduce (fn [ctx {:as _cd :exoscale.checkmate/keys [update]}]
            (cond-> ctx
              (ifn? update)
              update))
          ctx
          conditions))

(defn abort-ctx
  "Returns new context with abort cause, or nil, in which case we
  must allow retries to happen"
  [conditions ctx]
  (reduce (fn [_ {:as cd :exoscale.checkmate/keys [retry?]}]
            (when-not (retry? ctx)
              (reduced (assoc ctx :exoscale.checkmate/abort-cause cd))))
          nil
          conditions))
