(ns exoscale.checkmate.impl
  (:require [clojure.spec.alpha :as s]
            [clojure.tools.logging :as log]))

(def default-options
  {:exoscale.checkmate.hook/failure
   #(log/errorf "Run failure, aborting: runner=%s, run-id=%s, error=%s, retries=%s, abort-cause=%s"
                (:exoscale.checkmate/runner %)
                (:exoscale.checkmate/run-id %)
                (:exoscale.checkmate/error %)
                (:exoscale.checkmate/retries %)
                (-> % :exoscale.checkmate/abort-cause :exoscale.checkmate/id))
   :exoscale.checkmate.hook/error
   #(log/warnf "Run error, retrying: runner=%s, run-id=%s, error=%s, retries=%s"
                (:exoscale.checkmate/runner %)
                (:exoscale.checkmate/run-id %)
                (:exoscale.checkmate/error %)
                (:exoscale.checkmate/retries %))
   :exoscale.checkmate.hook/success identity})

(defn error-ctx
  [ctx err]
  (assoc ctx :exoscale.checkmate/error err))

(defn success-ctx
  [ctx x]
  (assoc ctx :exoscale.checkmate/success x))

(defn assert-conditions-valid! [conditions]
  (when (s/check-asserts?)
    (run! #(s/assert :exoscale.checkmate/condition %)
          conditions)))

(defn setup-conditions
  [conditions ctx runner]
  (reduce (fn [ctx {:as _cd :exoscale.checkmate/keys [setup]}]
            (cond-> ctx
              (ifn? setup)
              setup))
          (assoc ctx
                 :exoscale.checkmate/runner runner
                 :exoscale.checkmate/run-id (java.util.UUID/randomUUID)
                 :exoscale.checkmate/retries 0)
          conditions))

(defn update-conditions
  [conditions ctx]
  (reduce (fn [ctx {:as _cd :exoscale.checkmate/keys [update]}]
            (cond-> ctx
              (ifn? update)
              update))
          (update ctx :exoscale.checkmate/retries inc)
          conditions))

(defn abort-ctx
  "Returns new context with abort cause, or nil, in which case we
  must allow retries to happen"
  [conditions ctx]
  (if (seq conditions)
    (reduce (fn [_ {:as cd :exoscale.checkmate/keys [retry?]}]
              (when-not (retry? ctx)
                (reduced (assoc ctx :exoscale.checkmate/abort-cause cd))))
            nil
            conditions)
    ;; guard against empty conditions causing inf loop
    (assoc ctx
           :exoscale.checkmate/abort-cause
           {:exoscale.checkmate/id :exoscale.checkmate/empty-conditions})))
