(ns exoscale.checkmate.impl
  (:require [clojure.spec.alpha :as s]
            [clojure.tools.logging :as log]))

(def default-options
  {:exoscale.checkmate.hook/failure
   #(log/errorf "Run failure, aborting: runner=%s, run-id=%s, error=%s, retries=%s, abort-cause=%s"
                (:exoscale.checkmate/runner %)
                (:exoscale.checkmate/run-id %)
                (:exoscale.checkmate/error %)
                (:exoscale.checkmate/retries %)
                (-> % :exoscale.checkmate/abort-cause :exoscale.checkmate/id))
   :exoscale.checkmate.hook/error
   #(log/warnf "Run error, retrying: runner=%s, run-id=%s, error=%s, retries=%s"
                (:exoscale.checkmate/runner %)
                (:exoscale.checkmate/run-id %)
                (:exoscale.checkmate/error %)
                (:exoscale.checkmate/retries %))
   :exoscale.checkmate.hook/success identity
   :exoscale.checkmate/handler (fn [ctx x] (assoc ctx :exoscale.checkmate/success x))
   :exoscale.checkmate/return :exoscale.checkmate/success})

(defn into-queue
  [cds]
  (into clojure.lang.PersistentQueue/EMPTY
        cds))

(defn error-ctx
  [ctx err]
  (assoc ctx :exoscale.checkmate/error err))

(defn error?
  [ctx]
  (:exoscale.checkmate/error ctx))

(defn fail?
  [ctx]
  (:exoscale.checkmate/abort-cause ctx))

(defn init-ctx
  [ctx conditions]
  (assoc ctx :exoscale.checkmate/conditions conditions))

(defn assert-conditions-valid! [conditions]
  (when (s/check-asserts?)
    (run! #(s/assert :exoscale.checkmate/condition %)
          conditions)))

(defn -setup-conditions
  [{:as ctx :exoscale.checkmate/keys [setup-conditions]}]
  (if-let [{:exoscale.checkmate/keys [setup]} (peek setup-conditions)]
    (recur (cond-> (assoc ctx :exoscale.checkmate/setup-conditions (pop setup-conditions))
             (ifn? setup)
             setup))
    ctx))

(defn setup-conditions [{:as ctx :exoscale.checkmate/keys [conditions]} runner]
  (-> ctx
      (assoc
       :exoscale.checkmate/runner runner
       :exoscale.checkmate/run-id (java.util.UUID/randomUUID)
       :exoscale.checkmate/retries 0
       :exoscale.checkmate/setup-conditions (into-queue conditions))
      -setup-conditions))

(defn -update-conditions
  [{:as ctx :exoscale.checkmate/keys [update-conditions]}]
  (if-let [{:exoscale.checkmate/keys [update]} (peek update-conditions)]
    (recur (cond-> (assoc ctx
                          :exoscale.checkmate/update-conditions
                          (pop update-conditions))
             (ifn? update)
             update))
    ctx))

(defn update-conditions
  [{:as ctx :exoscale.checkmate/keys [conditions]}]
  (-> ctx
      (clojure.core/update :exoscale.checkmate/retries inc)
      (assoc :exoscale.checkmate/update-conditions (into-queue conditions))
      -update-conditions))

(defn -teardown-conditions
  [{:as ctx :exoscale.checkmate/keys [teardown-conditions]}]
  (if-let [{:exoscale.checkmate/keys [teardown]} (peek teardown-conditions)]
    (recur (cond-> (assoc ctx :exoscale.checkmate/teardown-conditions (pop teardown-conditions))
             (ifn? teardown)
             teardown))
    ctx))

(defn teardown-conditions
  [{:as ctx :exoscale.checkmate/keys [conditions]}]
  (-> ctx
      (assoc :exoscale.checkmate/teardown-conditions (into-queue conditions))
      -teardown-conditions))

(defn -abort-ctx
  "Returns new context with abort cause, or nil, in which case we
  must allow retries to happen"
  [{:as ctx :exoscale.checkmate/keys [abort-conditions]}]
  (if-let [{:as cd :exoscale.checkmate/keys [retry?]} (peek abort-conditions)]
    (if (retry? ctx)
      (recur (assoc ctx :exoscale.checkmate/abort-conditions (pop abort-conditions)))
      (assoc ctx
             :exoscale.checkmate/abort-cause cd
             :exoscale.checkmate/abort-conditions nil))
    ctx))

(defn abort-ctx
  [{:as ctx :exoscale.checkmate/keys [conditions]}]
  (if (seq conditions)
    (-> ctx
        (assoc :exoscale.checkmate/abort-conditions (into-queue conditions))
        -abort-ctx)
    (assoc ctx :exoscale.checkmate/abort-cause {:exoscale.checkmate/id :exoscale.checkmate/empty-conditions})))
