(ns seql.helpers
  "A collection of functions and macros used to help building
   SEQL schemas conforming to the spec provided in :seql.core/schema"
  (:require [clojure.spec.alpha :as s]
            [seql.schema        :as schema]
            [honey.sql.helpers  :as h]))

(defn- qualify
  [entity-name id]
  (keyword (format "%s/%s" (name entity-name) (name id))))

(defn- ensure-context!
  [expected forms]
  (doseq [{:keys [type context] :as form} (flatten forms)]
    (when (or (nil? type) (nil? context))
      (throw
       (ex-info "Unknown form encountered"
                {:type ::unknown-form
                 :form form})))
    (when-not (= context expected)
      (throw
       (ex-info (format "The %s form should be used inside %s definitions"
                        (if type (name type) "make-schema")
                        (if context (name context) "top-level"))
                {})))))

(defn ^:deprecated ident
  "Marks the current field as ident, used inside field definitions"
  []
  {:type :ident :context ::field})

(defn column-name
  [field colname]
  {:type :column-name :context ::entity :field field :colname colname})

(defn field
  "Define an entity field, with optional details.
   Possible detail functions include: `ident`"
  [id & details]
  (ensure-context! ::field details)
  (conj (mapv #(assoc % :field id :context ::entity) details)
        {:field id :type :field :context ::entity}))

(defn has-many
  "Express a one to many relation between the current
   entity and a remote one, expects a tuple of local
   ID to (qualified) remote ID."
  [field [local remote]]
  [{:type          :one-to-many
    :context       ::entity
    :field         field
    :remote-entity (keyword (namespace remote))
    :remote-id     remote
    :local-id      local}])

(defn has-one
  "Express a one to one relation between the current
   entity and a remote one, expects a tuple of local
   ID to (qualified) remote ID."
  [field [local remote]]
  [{:type          :one-to-one
    :context       ::entity
    :field         field
    :remote-entity (keyword (namespace remote))
    :remote-id     remote
    :local-id      local}])

(defn has-many-through
  ""
  [field [local left right remote]]
  [{:type               :many-to-many
    :context            ::entity
    :field              field
    :intermediate       (keyword (namespace left))
    :intermediate-left  left
    :intermediate-right right
    :remote-entity      (keyword (namespace remote))
    :remote-id          remote
    :local-id           local}])

(defn condition
  "Build a condition which can be used to filter results
   at the database layer.

   With a single arg, builds a condition bearing the name
   of a field to test pure equality.

   With two args, builds a condition testing equality
   against the provided field name.

   With three args, tests a field name against a provided value."
  ([field from-field value]
   {:type      :static-condition
    :context   ::entity
    :field      field
    :from-field from-field
    :value      value})
  ([field from-field]
   {:type       :field-condition
    :context   ::entity
    :field      field
    :from-field from-field})
  ([field]
   {:type       :field-condition
    :context   ::entity
    :field      field
    :from-field field}))

(defmacro inline-condition
  "Provide a function tail which shall yield a
   honeysql fragment to express a where condition on
   a field."
  [field args & body]
  {:type    :inline-condition
   :context ::entity
   :field   field
   :arity   (count args)
   :handler `(fn ~args ~@body)})

(defmacro mutation
  "Provide a function tail to perform a mutation against an entity"
  [mutation-name spec bindv & body]
  (when-not (and (vector? bindv)
                 (= 1 (count bindv)))
    (throw (ex-info "bad binding vector for mutation" {:bindv bindv})))
  {:type    :mutation
   :context ::entity
   :field   mutation-name
   :spec    spec
   :handler `(fn ~bindv ~@body)})

#_(s/def ::methods #{:create :update :delete})
#_(s/def ::args (s/cat :methods (s/* (s/or :plain ::methods :renamed (s/cat :method ::methods :name keyword?)))))
#_(when-not (s/valid? ::args args)
    (throw (ex-info "bad argumentsfor mutation" {:args args})))
#_(let [{:keys [methods]} (s/conform ::args args)])

(defn add-create-mutation
  [& [create-kw]]
  {:type    :create-mutation
   :context ::entity
   :spec    (or create-kw :create)})

(defn mutation-fn
  "Provide a function to perform a named mutation"
  [mutation-name spec handler]
  {:type    :mutation
   :context ::entity
   :field   mutation-name
   :spec    spec
   :handler handler})

(defmulti ^{:no-doc true}
  merge-entity-component
  (fn [_ v] (:type v)))

(defmethod merge-entity-component :field
  [{:keys [entity] :as schema} {:keys [field]}]
  (update schema :fields conj (qualify entity field)))

(defmethod merge-entity-component :column-name
  [{:keys [entity] :as schema} {:keys [field colname]}]
  (assoc-in schema [:overrides (qualify entity field)] colname))

(defmethod merge-entity-component :ident
  [schema _]
  schema)

(defmethod merge-entity-component :one-to-many
  [{:keys [entity] :as schema} {:keys [field] :as rel}]
  (update schema :relations
          assoc
          (qualify entity field)
          (-> rel
              (update :local-id (partial qualify entity))
              (dissoc :field :context))))

(defmethod merge-entity-component :one-to-one
  [{:keys [entity] :as schema} {:keys [field] :as rel}]
  (update schema :relations
          assoc
          (qualify entity field)
          (-> rel
              (update :local-id (partial qualify entity))
              (dissoc :field :context))))

(defmethod merge-entity-component :many-to-many
  [{:keys [entity] :as schema} {:keys [field] :as rel}]
  (update schema :relations
          assoc
          (qualify entity field)
          (-> rel
              (update :local-id (partial qualify entity))
              (dissoc :field :context))))

(defmethod merge-entity-component :static-condition
  [{:keys [entity] :as schema} {:keys [field from-field value]}]
  (update schema :conditions assoc
          (qualify entity field)
          {:type :static
           :field (qualify entity from-field)
           :value value}))

(defmethod merge-entity-component :field-condition
  [{:keys [entity] :as schema} {:keys [field from-field]}]
  (update schema :conditions assoc
          (qualify entity field)
          {:type :field
           :field (qualify entity from-field)}))

(defmethod merge-entity-component :inline-condition
  [{:keys [entity] :as schema} {:keys [field arity handler]}]
  (update schema :conditions assoc
          (qualify entity field)
          {:type    :inline
           :arity   arity
           :handler handler}))

(defmethod merge-entity-component :mutation
  [{:keys [entity] :as schema} {:keys [field spec handler]}]
  (update schema :mutations assoc
          (qualify entity field)
          {:spec spec :handler handler}))

(defmethod merge-entity-component :create-mutation
  [{:keys [table entity] :as schema} {:keys [spec]}]
  (let [mname (qualify entity spec)
        handler (fn [params]
                  (h/insert-into table (h/values [(schema/as-row schema entity params)])))]
    (update schema :mutations assoc
            mname
            {:spec mname :handler handler})))

(defn- keys-spec-fields
  "Resolve qualified key belonging to an `s/keys` spec."
  [spec-name]
  (let [spec-def (s/describe spec-name)]
    (when (and (coll? spec-def) (= 'keys (first spec-def)))
      (into []
            (comp (drop 1)
                  (partition-all 2)
                  (filter #(contains? #{:req :opt} (first %)))
                  (mapcat last))
            spec-def))))

(defn from-spec
  ""
  [arg & components]
  (ensure-context! ::entity (flatten components))
  (let [[entity-name table-name] (if (sequential? arg) arg [arg])]
    (reduce merge-entity-component
            {:entity  (keyword (cond-> entity-name (qualified-keyword? entity-name) namespace))
             :table   (or table-name (keyword (name entity-name)))
             :spec    entity-name
             :context ::schema
             :type    :entity
             :fields  (keys-spec-fields entity-name)}
            (flatten components))))

(defn entity
  "Provide an entity description. Expects either a keyword or a tuple
   of `[entity-name table-name]` and a list of details as expressed by
   `field`, `has-many`, `condition`, and `mutation`"
  [arg & components]
  (let [components (flatten components)]
    (ensure-context! ::entity components)
    (let [[entity-name table-name] (if (sequential? arg) arg [arg])]
      (reduce merge-entity-component
              {:entity     (keyword (cond-> entity-name (qualified-keyword? entity-name) namespace))
               :table      (or table-name (keyword (name entity-name)))
               :context    ::schema
               :type       :entity
               :fields     []}
              components))))

(defn make-schema
  "Provide a complete schema of entites"
  [& entities]
  (ensure-context! ::schema entities)
  (into {} (map (juxt :entity #(dissoc % :entity :context :type)) entities)))
