(ns exoscale.specs
  "Utilities extending clj.specs with missing bits (some of which will
  come with alpha2), like metadata, docstring support"
  (:refer-clojure :exclude [meta])
  (:require [clojure.spec.alpha :as s]))

;;; Metadata support
;;; The following only works only for registered specs

(defonce metadata-registry (atom {}))
(s/def ::metadata-registry-val (s/map-of qualified-keyword? any?))

(s/fdef vary-meta!
  :args (s/cat :k qualified-keyword?
               :f ifn?
               :args (s/* any?))
  :ret qualified-keyword?)
(defn vary-meta!
  "Like clojure.core/vary-meta but for registered specs, mutates the
  meta in place, return the keyword spec"
  [k f & args]
  (swap! metadata-registry
         #(update % k
                  (fn [m]
                    (apply f m args))))
  k)

(s/fdef with-meta!
  :args (s/cat :k qualified-keyword?
               :meta any?)
  :ret ::metadata-registry-val)
(defn with-meta!
  "Like clojure.core/with-meta but for registered specs, mutates the
  meta in place, return the keyword spec"
  [k m]
  (swap! metadata-registry
         #(assoc % k m))
  k)

(s/fdef meta
  :args (s/cat :k qualified-keyword?)
  :ret any?)
(defn meta
  "Like clojure.core/meta but for registered specs."
  [k]
  (get @metadata-registry k))

(s/fdef unregister-meta!
  :args (s/cat :k qualified-keyword?)
  :ret qualified-keyword?)
(defn unregister-meta!
  "Unregister meta data for a spec"
  [k]
  (swap! metadata-registry dissoc k)
  k)

(s/fdef with-doc
  :args (s/cat :k qualified-keyword?
               :doc string?)
  :ret qualified-keyword?)
(defn with-doc
  "Add doc metadata on a registered spec"
  [k doc]
  (vary-meta! k assoc :doc doc))

(s/fdef doc
  :args (s/cat :k qualified-keyword?)
  :ret (s/nilable string?))
(defn doc
  "Returns doc associated with spec"
  [k]
  (some-> k meta :doc))

(s/def ::def-args
  (s/cat :spec-key qualified-keyword?
         :meta (s/? map?)
         :docstring (s/? string?)
         :spec any?))

(s/fdef exoscale.specs/def
  :args ::def-args
  :ret any?)
(defmacro def
  "Same as `clojure.spec/def` but take an optional `docstring`, `attr-map`.

  ;; just a doctring:

  (exoscale.specs/def ::foo
    \"this is a foo\"
     string?)

  ;; add metadata
  (exoscale.specs/def ::foo
    \"this is a foo\"
    {:something :interesting}
    string?)

  It's also identical to `clojure.spec/def` with no options (as convenience):
  (exoscale.specs/def ::foo string?)

  It returns the spec key, just like `clojure.spec/def`"
  {:arglists '([spec-key spec]
               [spec-key doc-string spec]
               [spec-key metadata spec]
               [spec-key metadata doc-string spec])}
  [& args]
  (let [{:keys [spec-key docstring meta spec]} (s/conform ::def-args args)]
    `(do
       (s/def ~spec-key ~spec)
       ~@(cond-> []
           docstring
           (conj `(with-doc ~spec-key ~docstring))
           meta
           (conj `(vary-meta! ~spec-key merge ~meta)))
       ~spec-key)))
