(ns exoscale.specs.net
  (:require [clojure.spec.alpha :as s]
            [clojure.spec.gen.alpha :as gen]
            [clojure.test.check.generators]
            [clojure.string :as str])
  #?(:clj (:import (exoscale.specs.validator InetAddressValidator))
     :cljs (:import (goog.Uri))))

#?(:clj (def ^:private ^InetAddressValidator validator
          (InetAddressValidator/getInstance)))

(s/def ::ipv4 (s/with-gen (s/and string?
                                 #?(:clj #(.isValidInet4Address validator %)
                                    :cljs #(re-find #"^(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)\.(25[0-5]|2[0-4][0-9]|[01]?[0-9][0-9]?)$"
                                                    %)))
                #(gen/fmap (fn [x] (str/join "." x))
                           (gen/tuple (gen/return 159)
                                      (gen/large-integer* {:min 0 :max 255})
                                      (gen/large-integer* {:min 0 :max 255})
                                      (gen/large-integer* {:min 0 :max 255})))))

(s/def ::ipv6 (s/with-gen (s/and string?
                                 #?(:clj #(.isValidInet6Address validator %)
                                    :cljs #(re-find #"(?:^|(?<=\s))(([0-9a-fA-F]{1,4}:){7,7}[0-9a-fA-F]{1,4}|([0-9a-fA-F]{1,4}:){1,7}:|([0-9a-fA-F]{1,4}:){1,6}:[0-9a-fA-F]{1,4}|([0-9a-fA-F]{1,4}:){1,5}(:[0-9a-fA-F]{1,4}){1,2}|([0-9a-fA-F]{1,4}:){1,4}(:[0-9a-fA-F]{1,4}){1,3}|([0-9a-fA-F]{1,4}:){1,3}(:[0-9a-fA-F]{1,4}){1,4}|([0-9a-fA-F]{1,4}:){1,2}(:[0-9a-fA-F]{1,4}){1,5}|[0-9a-fA-F]{1,4}:((:[0-9a-fA-F]{1,4}){1,6})|:((:[0-9a-fA-F]{1,4}){1,7}|:)|fe80:(:[0-9a-fA-F]{0,4}){0,4}%[0-9a-zA-Z]{1,}|::(ffff(:0{1,4}){0,1}:){0,1}((25[0-5]|(2[0-4]|1{0,1}[0-9]){0,1}[0-9])\.){3,3}(25[0-5]|(2[0-4]|1{0,1}[0-9]){0,1}[0-9])|([0-9a-fA-F]{1,4}:){1,4}:((25[0-5]|(2[0-4]|1{0,1}[0-9]){0,1}[0-9])\.){3,3}(25[0-5]|(2[0-4]|1{0,1}[0-9]){0,1}[0-9]))(?=\s|$)"
                                                    %)))
                #(s/gen #{"1:2:3:4:5:6:7:8" "::ffff:10.0.0.1" "::ffff:1.2.3.4"})))

(s/def ::ip (s/or :ipv4 ::ipv4
                  :ipv6 ::ipv6))

(s/def ::mac-address
  (s/and string?
         #(re-matches #"([0-9a-fA-F]{2}:??){5}([0-9a-fA-F]{2})" %)))

(s/def ::port (s/int-in 0 65354))

(s/def ::url
  (s/and string?
         #(try
            (->> #?(:clj (java.net.URI. %)
                    :cljs (goog.Uri. %))
                 (.getScheme)
                 str/lower-case
                 (contains? #{"https" "http"}))
            (catch #?(:clj Exception :cljs :default) _))))

(defn- split-cidr
  [cidr]
  (let [parts (str/split cidr #"/")]
    (when (-> parts count (= 2))
      parts)))

(defn- valid-ipv4-ip-and-cidr?
  [[ip cidr]]
  (and (s/valid? ::ipv4 ip)
       (s/valid? #(when-let [block (parse-long %)]
                    (<= 0 block 32))
                 cidr)))

(defn- valid-ipv6-ip-and-cidr?
  [[ip cidr]]
  (and (s/valid? ::ipv6 ip)
       (s/valid? #(when-let [block (parse-long %)]
                    (<= 0 block 128))
                 cidr)))

(s/def ::cidr-ipv4
  (s/with-gen (s/and string?
                     #(->> %
                           (split-cidr)
                           (valid-ipv4-ip-and-cidr?)))
    #(gen/fmap (fn [x] (str (str/join "." x) "/32"))
               (gen/tuple (gen/return 159)
                          (gen/large-integer* {:min 0 :max 255})
                          (gen/large-integer* {:min 0 :max 255})
                          (gen/large-integer* {:min 0 :max 255})))))

(s/def ::cidr-ipv6
  (s/with-gen (s/and string?
                     #(->> %
                           (split-cidr)
                           (valid-ipv6-ip-and-cidr?)))
    #(s/gen #{"1:2:3:4:5:6:7:8/96" "::ffff:10.0.0.1/96" "::ffff:1.2.3.4/96"})))

(s/def ::cidr (s/or :cidr-ipv4 ::cidr-ipv4
                    :cidr-ipv6 ::cidr-ipv6))
