(ns exoscale.specs.time
  (:require [exoscale.coax :as cx]
            [clojure.spec.alpha :as s]))

(def second-ms 1000)
(def minute-ms (* second-ms 60))
(def hour-ms (* minute-ms 60))
(def day-ms (* 24 hour-ms))
(def week-ms (* 7 day-ms))
(def month-ms (* 30 day-ms))
(def year-ms (* 12 month-ms))

(defn to-ms [unit x]
  (* x
     (case unit
       "ms" 1
       "s" second-ms
       "m" minute-ms
       "h" hour-ms
       "d" day-ms
       "w" week-ms
       "M" month-ms
       "Y" year-ms)))

(defn duration
  "Turns a number/string to a duration if possible.

  Will take a string in the form of:
  - 1ms
  - 1h30s

  units supported are:
  - ms milliseconds
  - s  seconds
  - m  minutes
  - h  hours
  - d  days
  - w  weeks
  - M  months
  - Y  years

  Numbers will be assumed to be milliseconds"
  [x]
  (letfn [(ex! [x e]
            (throw (ex-info (str "Invalid duration value "  x)
                            {:val x
                             :exoscale.ex/type :exoscale.ex/incorrect}
                            e)))
          (from-string [x]
            (or (some->> x
                         (re-seq #"([0-9]+)([mshdwMY]+)")
                         (reduce (fn [d [_ x u]]
                                   (+ d (to-ms u
                                               (#?(:clj Long/parseLong :cljs js/parseInt) x))))
                                 0))
                (ex! x nil)))]
    (try
      (cond
        (string? x)
        (or (from-string x)
            (ex! x nil))

        (nat-int? x) x
        :else (ex! x nil))
      (catch #?(:clj Exception :cljs :default) e
        (ex! x e)))))

(s/def ::duration
  (-> #(try (duration %)
            (catch #?(:clj Exception :cljs :default) _e
              false))
      (s/with-gen #(s/gen nat-int?))))

(cx/def ::duration
  (fn [x _opts]
    (try (duration x)
         (catch #?(:clj Exception :cljs :default) _e
           :exoscale.coax/invalid))))
