package jre2;

import java.util.ArrayList;
import java.util.List;


public class JRE2 implements java.io.Closeable {
	private long nativeRE2ptr;
	private String regex;


	public JRE2(String regex) throws JRE2Exception {
		nativeRE2ptr = compile(regex.getBytes(StringUTF8.UTF8_CHARSET), -1);
		this.regex = regex;
	}

	public JRE2(String regex, int maxMem) throws JRE2Exception {
		nativeRE2ptr = compile(regex.getBytes(StringUTF8.UTF8_CHARSET), maxMem);
		this.regex = regex;
	}


	public String toString() {
		return regex;
	}


	/**
	 * Frees the native RE2 object.
	 */
	public void close() {
		if (nativeRE2ptr != 0) {
			free(nativeRE2ptr);
			nativeRE2ptr = 0;
		}
	}


	public Matcher matcher(String input) {
		return new Matcher(this, input);
	}

	public Matcher matcher(StringUTF8 inputUTF8) {
		return new Matcher(this, inputUTF8);
	}


	/**
	 * Splits the given input StringUTF8 around matches of this pattern.
	 *
	 * Same behavior as (and derived from the code for) java.util.regex.Pattern.split()
	 * except that input and outputs are StringUTF8 instances.
	 */
    public StringUTF8[] split(StringUTF8 input, int limit) throws JRE2Exception {
        int index = 0;
        boolean matchLimited = limit > 0;
        List<StringUTF8> matchList = new ArrayList<StringUTF8>();
        Matcher m = matcher(input);

        // Add segments before each match found
        while(m.find()) {
            if (!matchLimited || matchList.size() < limit - 1) {
                StringUTF8 match = new StringUTF8(input, index, m.start() - index);
                matchList.add(match);
                index = m.end();
            } else if (matchList.size() == limit - 1) { // last one
                StringUTF8 match = new StringUTF8(input, index, input.length() - index);
                matchList.add(match);
                index = m.end();
            }
        }

        // If no match was found, return input
        if (index == 0)
            return new StringUTF8[] {input};

        // Add remaining segment
        if (!matchLimited || matchList.size() < limit)
            matchList.add(new StringUTF8(input, index, input.length() - index));

        // Construct result
        int resultSize = matchList.size();
        if (limit == 0)
            while (resultSize > 0 && matchList.get(resultSize-1).equals(""))
                resultSize--;
        StringUTF8[] result = new StringUTF8[resultSize];
        return matchList.subList(0, resultSize).toArray(result);
    }


//-----------------------------------------------------------------------------------------------
	// These methods are not public but are still accessible by Matcher.

	// Returns group indices with origin at position 0 of inputUTF8.bytes.
	int[] fullMatch(StringUTF8 inputUTF8) throws JRE2Exception {
		if (nativeRE2ptr == 0) {
			throw new IllegalStateException("Attempt to match with freed RE2 instance");
		}
		return fullMatch(nativeRE2ptr, inputUTF8.bytes(), inputUTF8.offset(), inputUTF8.length());
	}

	// Returns group indices with origin at position 0 of inputUTF8.bytes.
	int[] partialMatch(StringUTF8 inputUTF8, int offset) throws JRE2Exception {
		if (nativeRE2ptr == 0) {
			throw new IllegalStateException("Attempt to match with freed RE2 instance");
		}
		return partialMatch(nativeRE2ptr, inputUTF8.bytes(), inputUTF8.offset() + offset, inputUTF8.length() - offset);
	}

	int groupCount() {
		if (nativeRE2ptr == 0) {
			throw new IllegalStateException("Attempt to find group count of freed RE2 instance");
		}
		return groupCount(nativeRE2ptr);

	}
//-----------------------------------------------------------------------------------------------


	private static native long compile(byte[] regexUTF8, int maxMem) throws JRE2Exception;
	private static native int[] fullMatch(long RE2ptr, byte[] inputUTF8, int offset, int length) throws JRE2Exception;
	private static native int[] partialMatch(long RE2ptr, byte[] inputUTF8, int offset, int length) throws JRE2Exception;
	private static native void free(long RE2ptr);
	private static native int groupCount(long RE2ptr);

	static {
		System.loadLibrary("JRE2");
	}

	public static void main(String[] args) throws JRE2Exception {
		System.out.println(new JRE2("a.b").split(new StringUTF8("zzzacbwww"), 0));
	}
}
