(ns farbetter.chumbawamba.websocket
  (:require
   [#?(:clj clojure.core.async :cljs cljs.core.async)
    :refer [<! timeout]]
   #?(:clj [gniazdo.core :as ws])
   [farbetter.utils :as u :refer
    [throw-far-error #?@(:clj [go-safe inspect sym-map])]]
   [taoensso.timbre :as timbre
    #?(:clj :refer :cljs :refer-macros) [debugf errorf infof warnf]])
  #?(:cljs
     (:require-macros
      [farbetter.utils :refer [go-safe inspect sym-map]])
     :clj
     (:import [java.util Arrays])))

#?(:cljs (def Exception js/Error))
(def wait-time-ms 5)
(def ws-uri-pattern (re-pattern "^(ws|wss)://.+"))

(defprotocol IWebsocket
  (ws-send [this string-or-bytes])
  (ws-close [this]))

#?(:clj
   (defn make-websocket-impl-clj [uri on-close on-rcv]
     (go-safe
      (try
        (let [on-bin (fn [bytes offset length]
                       (on-rcv (Arrays/copyOfRange bytes offset
                                                    (+ offset length))))
               socket (ws/connect uri
                                  :on-binary on-bin
                                  :on-receive on-rcv
                                  :on-close (fn [status reason]
                                              (on-close reason)))]
           [:success socket])
         (catch java.net.ConnectException e
           [:failure :connection-failure]))))

   :cljs
   (do
     (defn make-websocket-impl-node [uri on-close on-rcv]
       (go-safe
         (try
           (let [JSWebSocket (js/require "ws")
                 socket (JSWebSocket. uri)
                 msg-handler (fn [data flags]
                               (if (.-binary flags)
                                 ;; convert to byte-array
                                 (on-rcv (.from js/Int8Array data))
                                 (on-rcv data)))]
             (.on socket "message" msg-handler)
             (.on socket "close" (fn [] (on-close :closed)))
             (loop []
               (condp = (.-readyState socket)
                 0 (do
                     (<! (timeout wait-time-ms))
                     (recur))
                 1 [:success socket]
                 [:failure :connection-failure])))
           (catch js/Error e
             (errorf (str "js/Error. Msg:" (.-message e)))
             [:failure {:error e}]))))

     (defn make-websocket-impl-ios [uri on-close on-rcv]
       (go-safe
         (try
           (let [socket (.makeWithURIOnCloseOnReceive js/FarWebSocket uri
                                                      on-close on-rcv)]
             (loop []
               (let [state (.getState socket)]
                 (condp = state
                   0 (do
                       (<! (timeout wait-time-ms))
                       (recur))
                   1 [:success socket]
                   [:failure :connection-failure]))))
           (catch js/Error e
             (errorf (str "js/Error. Msg:" (.-message e)))
             [:failure {:exception e}]))))

     (defn make-websocket-impl-browser [uri on-close on-rcv]
       (go-safe
         (let [socket (js/WebSocket. uri)]
           (set! (.-onmessage socket) (fn [msg] (on-rcv (.-data msg))))
           (set! (.-onclose socket) (fn [msg] (on-close :closed)))
           (set! (.-binaryType socket) "arraybuffer")
           (loop []
             (condp = (.-readyState socket)
               0 (do
                   (<! (timeout wait-time-ms))
                   (recur))
               1 [:success socket]
               [:failure :connection-failure])))))))

(defrecord Websocket [socket]
  IWebsocket
  (ws-send [this string-or-bytes]
    (#?(:clj ws/send-msg :cljs .send) socket string-or-bytes))

  (ws-close [this]
    #?(:clj (ws/close socket) :cljs (.close socket))))

(defn make-websocket [uri on-close on-rcv]
  (when-not (re-find ws-uri-pattern uri)
    (throw-far-error "URI must begin with ws:// or wss://"
                     :illegal-argument :illegal-uri (sym-map uri)))
  (go-safe
   (let [platform (u/get-platform-kw)
         f #?(:clj make-websocket-impl-clj
              :cljs (case platform
                      :node make-websocket-impl-node
                      :ios make-websocket-impl-ios
                      :browser make-websocket-impl-browser))
         [status ret] (<! (f uri on-close on-rcv))]
     (if (= :success status)
       [status (->Websocket ret)]
       [status ret]))))
