(ns freedomdb.row-store
  (:require
   [farbetter.utils :refer [#?@(:clj [inspect sym-map]) throw-far-error]]
   [freedomdb.schemas :refer
    [DBTableMetadataType DBType FieldAttrsMapType FieldNameType FieldsMapType
     FieldTypeType GetAllRowsOutputStyleType GetAllRowsReturnType
     InequalityOperatorType RowIdType TableNameType TypeMapType ValueMapType
     ValueType]]
   [schema.core :as s :include-macros true]))

(defprotocol RowStore
  (create-table- [this table-name fields-map])
  (drop-table- [this table-name])
  (get-metadata- [this table-name])
  (get-table-names-set- [this])
  (add-field- [this table-name field-name attrs])
  (remove-field- [this table-name field-name])
  (modify-field- [this table-name field-name attrs])
  (get-row-count- [this table-name])
  (get-row- [this table-name row-id])
  (get-all-rows- [this table-name output-fmt])
  (get-row-ids-eq- [this table-name field-name v v-type])
  (get-row-ids-ineq- [this table-name field-name v v-type op])
  (put-row- [this table-name val-map type-map indexed-fields])
  (update-row- [this table-name row-id old-val-map new-val-map
                type-map indexed-fields])
  (delete-row- [this table-name row-id type-map])
  (delete-all-rows- [this table-name type-map]))

(s/defn get-metadata :- (s/maybe DBTableMetadataType)
  [db :- DBType
   table-name :- TableNameType]
  (get-metadata- db table-name))

(s/defn create-table :- DBType
  [db :- DBType
   table-name :- TableNameType
   fields-map :- FieldsMapType]
  (create-table- db table-name fields-map))

(s/defn drop-table
  [db :- DBType
   table-name :- TableNameType]
  (drop-table- db table-name))

(s/defn get-table-names-set :- #{TableNameType}
  [db :- DBType]
  (get-table-names-set- db))

(s/defn add-field :- DBType
  [db :- DBType
   table-name :- TableNameType
   field-name :- FieldNameType
   attrs :- FieldAttrsMapType]
  (add-field- db table-name field-name attrs))

(s/defn remove-field :- DBType
  [db :- DBType
   table-name :- TableNameType
   field-name :- FieldNameType]
  (remove-field- db table-name field-name))

(s/defn modify-field :- DBType
  [db :- DBType
   table-name :- TableNameType
   field-name :- FieldNameType
   attrs :- FieldAttrsMapType]
  (modify-field- db table-name field-name attrs))

(s/defn get-row-count :- s/Num
  [db :- DBType
   table-name :- TableNameType]
  (get-row-count- db table-name))

(s/defn get-row :- ValueMapType
  [db :- DBType
   table-name :- TableNameType
   row-id :- RowIdType]
  (get-row- db table-name row-id))

(s/defn get-all-rows :- GetAllRowsReturnType
  [db :- DBType
   table-name :- TableNameType
   output-fmt :- GetAllRowsOutputStyleType]
  (get-all-rows- db table-name output-fmt))

(s/defn get-row-ids-eq :- (s/maybe #{RowIdType})
  [db :- DBType
   table-name :- TableNameType
   field-name :- FieldNameType
   v :- ValueType
   v-type :- FieldTypeType]
  (get-row-ids-eq- db table-name field-name v v-type))

(s/defn get-row-ids-ineq :- (s/maybe #{RowIdType})
  [db :- DBType
   table-name :- TableNameType
   field-name :- FieldNameType
   v :- ValueType
   v-type :- FieldTypeType
   op :- InequalityOperatorType]
  (get-row-ids-ineq- db table-name field-name v v-type op))

(s/defn put-row :- DBType
  [db :- DBType
   table-name :- TableNameType
   val-map :- ValueMapType
   type-map :- TypeMapType
   indexed-fields :- #{FieldNameType}]
  (put-row- db table-name val-map type-map indexed-fields))

(s/defn update-row :- DBType
  [db :- DBType
   table-name :- TableNameType
   row-id :- RowIdType
   old-val-map :- ValueMapType
   new-val-map :- ValueMapType
   type-map :- TypeMapType
   indexed-fields :- #{FieldNameType}]
  (update-row- db table-name row-id old-val-map new-val-map
               type-map indexed-fields))

(s/defn delete-row :- DBType
  [db :- DBType
   table-name :- TableNameType
   row-id :- RowIdType
   type-map :- TypeMapType]
  (delete-row- db table-name row-id type-map))

(s/defn delete-all-rows :- DBType
  [db :- DBType
   table-name :- TableNameType
   type-map :- TypeMapType]
  (delete-all-rows- db table-name type-map))
