(ns freedomdb.durable-row-store
  (:require
   [clojure.set :refer [union]]
   [clojure.string :refer [join split]]
   [farbetter.utils :as u :refer
    [#?@(:clj [inspect sym-map]) edn->transit throw-far-error transit->edn]]
   [freedomdb.kv-store :as kv]
   [freedomdb.mem-avl-kv-store :as maks]
   [freedomdb.row-store :as rs :refer [RowStore]]
   [schema.core :as s :include-macros true])
  #?(:cljs
     (:require-macros
      [farbetter.utils :refer [inspect sym-map]])))

(declare add-to-vr-index get-all-fields get-defaults get-indexed-fields
         get-new-row-id get-type-map get-vr-row-ids int4->str
         make-durable-row-store make-rv-key make-vr-key parse-rv-key
         remove-val-from-vr-index store-row str->int4 update-vr-index)

(def separator "+")
(def separator-pattern #"\+")

(defrecord DurableRowStore [kv-store]
  RowStore
  (get-metadata- [this table-name]
    (let [k (name table-name)
          fields-map (transit->edn (kv/get kv-store :metadata k))]
      (if-not (seq fields-map)
        nil
        (let [all-fields (get-all-fields fields-map)
              indexed-fields (get-indexed-fields fields-map)
              type-map (get-type-map fields-map)
              defaults (get-defaults fields-map)]
          (sym-map table-name all-fields indexed-fields type-map defaults)))))

  (create-table- [this table-name fields-map]
    (let [k (name table-name)]
      (reduce-kv (fn [_ field-name field-attrs]
                   (when (= :any (:type field-attrs))
                     (throw-far-error
                      (str "Field type `:any` is not supported by durable "
                           "row stores.")
                      :illegal-argument :unsupported-field-type {})))
                 nil fields-map)
      (assoc this :kv-store
             (kv/put kv-store :metadata k (edn->transit fields-map)))))

  (drop-table- [this table-name]
    (let [metadata (rs/get-metadata- this table-name)
          {:keys [type-map]} metadata]
      (-> this
          (rs/delete-all-rows- table-name type-map)
          (update :kv-store kv/remove :metadata (name table-name)))))

  (get-table-names-set- [this]
    (reduce (fn [names-set table-name]
              (conj names-set (keyword table-name)))
            #{} (kv/keys kv-store :metadata)))

  ;; TODO: Implement these
  (add-field- [this table-name field-name attrs]
    this)

  (remove-field- [this table-name field-name]
    this)

  (modify-field- [this table-name field-name attrs]
    this)

  (get-row-count- [this table-name]
    (throw-far-error "get-row-count- is not implemented."
                     :execution-error :not-implemented {}))

  (get-row- [this table-name row-id]
    (let [rv-key (make-rv-key table-name row-id)]
      (-> (kv/get kv-store :rvi rv-key)
          transit->edn)))

  (get-all-rows- [this table-name output]
    (let [options {:prefix (make-rv-key table-name nil)
                   :output (case output
                             :row-ids-only :keys-only
                             :value-maps-only :values-only
                             :row-ids-and-value-maps :keys-and-values)}
          get-row-id #(:row-id (parse-rv-key %))
          rows (kv/filter kv-store :rvi options)
          get-value-map transit->edn
          xform (case output
                  :row-ids-only get-row-id
                  :value-maps-only get-value-map
                  :row-ids-and-value-maps (fn [[k v]]
                                            (let [row-id (get-row-id k)
                                                  val-map (get-value-map v)]
                                              [row-id val-map])))]
      (map xform rows)))

  (get-row-ids-eq- [this table-name field-name v v-type]
    (get-vr-row-ids kv-store (make-vr-key table-name field-name v v-type)))

  (get-row-ids-ineq- [this table-name field-name v v-type op]
    (let [options {:output :values-only
                   :prefix (make-vr-key table-name field-name nil nil)
                   :filter-op op
                   :filter-val (make-vr-key table-name field-name v v-type)}
          transit-sets (kv/filter kv-store :vri options)]
      (reduce (fn [output-set transit-set]
                (union output-set (transit->edn transit-set)))
              #{} transit-sets)))

  (put-row- [this table-name val-map type-map indexed-fields]
    (let [row-id (get-new-row-id)]
      (assoc this :kv-store
             (-> (store-row kv-store table-name row-id val-map)
                 (add-to-vr-index table-name row-id
                                  val-map type-map indexed-fields)))))

  (update-row- [this table-name row-id old-val-map set-val-map
                type-map indexed-fields]
    (let [merged-val-map (merge old-val-map set-val-map)]
      (assoc this :kv-store
             (-> (store-row kv-store table-name row-id merged-val-map)
                 (update-vr-index table-name row-id old-val-map
                                  set-val-map type-map indexed-fields)))))

  (delete-row- [this table-name row-id type-map]
    (let [val-map (rs/get-row- this table-name row-id)
          kv-store (kv/remove kv-store :rvi (make-rv-key table-name row-id))]
      (assoc this :kv-store
             (reduce-kv (fn [kv-store field-name v]
                          (let [v-type (type-map field-name)]
                            (remove-val-from-vr-index
                             kv-store table-name field-name v v-type row-id)))
                        kv-store val-map))))

  (delete-all-rows- [this table-name type-map]
    (let [rids (rs/get-all-rows- this table-name :row-ids-only)]
      (reduce (fn [db rid]
                (rs/delete-row- db table-name rid type-map))
              this rids))))

;;;;;;;;; Constructor ;;;;;;;;;

(defn make-durable-row-store [kv-store-type]
  (case kv-store-type
    :mem (->DurableRowStore (maks/make-mem-avl-kv-store))
    (throw-far-error (str "Unsupported KV Store type `" kv-store-type "`")
                     :illegal-argument :unsupported-kv-store-type
                     (sym-map kv-store-type))))

;;;;;;;;; Helper fns ;;;;;;;;;

(defn- store-row [kv-store table-name row-id val-map]
  (let [rv-key (make-rv-key table-name row-id)]
    (kv/put kv-store :rvi rv-key (edn->transit val-map))))


(defn- make-rv-key [table-name row-id]
  (join separator [(name table-name) row-id]))

(defn- make-vr-key [table-name field-name v v-type]
  (let [fmt-fn (case v-type
                 :int4 int4->str
                 :str1000 identity
                 :bool str
                 :kw name
                 nil identity)]
    (join separator [(name table-name) (name field-name) (fmt-fn v)])))

(defn- parse-rv-key [rv-key]
  (let [[table-name row-id] (split rv-key separator-pattern)
        row-id (#?(:cljs js/parseInt :clj Long/parseLong s) row-id)]
    (sym-map table-name row-id)))

(defn- get-vr-row-ids [kv-store vr-key]
  (let [row-ids-transit-str (kv/get kv-store :vri vr-key)]
    (transit->edn row-ids-transit-str)))

(defn- put-vr-row-ids [kv-store vr-key row-ids]
  (->> (edn->transit row-ids)
       (kv/put kv-store :vri vr-key)))

(defn- add-val-to-vr-index [kv-store table-name field-name v v-type row-id
                            indexed-fields]
  (if-not (contains? indexed-fields field-name)
    kv-store
    (let [vr-key (make-vr-key table-name field-name v v-type)
          old-row-ids (or (get-vr-row-ids kv-store vr-key)
                          #{})
          new-row-ids (conj old-row-ids row-id)]
      (put-vr-row-ids kv-store vr-key new-row-ids))))

(defn- remove-val-from-vr-index [kv-store table-name field-name v v-type row-id]
  (let [vr-key (make-vr-key table-name field-name v v-type)
        old-row-ids (get-vr-row-ids kv-store vr-key)
        new-row-ids (disj old-row-ids row-id)]
    (if (seq new-row-ids)
      (put-vr-row-ids kv-store vr-key new-row-ids)
      (kv/remove kv-store :vri vr-key))))

(defn- add-to-vr-index [kv-store table-name row-id val-map type-map
                        indexed-fields]
  (reduce-kv (fn [kv-store field-name v]
               (add-val-to-vr-index kv-store table-name field-name
                                    v (type-map field-name) row-id
                                    indexed-fields))
             kv-store val-map))

(defn- update-vr-index [kv-store table-name row-id old-val-map
                        set-val-map type-map indexed-fields]
  (let [f (fn [kv-store field-name new-val]
            (let [old-val (old-val-map field-name)
                  val-type (type-map field-name)]
              (-> kv-store
                  (remove-val-from-vr-index
                   table-name field-name old-val val-type row-id)
                  (add-val-to-vr-index
                   table-name field-name new-val val-type row-id
                   indexed-fields))))]
    (reduce-kv f kv-store set-val-map)))

;;TODO: Replace this with something in the kv-store
(defonce next-row-id (atom 0))

(defn- get-new-row-id []
  (swap! next-row-id inc))

(defn- get-all-fields [fields-map]
  (set (keys fields-map)))

(defn- get-indexed-fields [fields-map]
  (reduce-kv (fn [acc k v]
               (if (:indexed v)
                 (conj acc k)
                 acc))
             #{} fields-map))

(defn- get-defaults [fields-map]
  (reduce-kv (fn [acc k v]
               (let [{:keys [default]} v]
                 (if (nil? default)
                   acc
                   (assoc acc k default))))
             {} fields-map))

(defn- get-type-map [fields-map]
  (reduce-kv (fn [acc k v]
               (assoc acc k (:type v)))
             {} fields-map))

;;;;;;;;; String encoding / decoding ;;;;;;;;;

(def offset 2147483648)
(def initial-divisor 1000000000)
(def ascii-zero 48)

#?(:clj
   (defn int4->str
     "Encode an int4 as a string, ensuring lexicographical ordering."
     [int4]
     (let [shifted-num (+ offset int4)]
       (format "%010d" shifted-num)))

   :cljs
   ;; goog.string.format is about 4x slower than this fn
   (defn int4->str
     "Encode an int4 as a string, ensuring lexicographical ordering."
     [int4]
     (let [shifted-num (+ offset int4)]
       (loop [num shifted-num
              divisor initial-divisor
              output []]
         (let [digit (quot num divisor)]
           (if (< divisor 1)
             (clojure.string/join output)
             (recur (- num (* digit divisor))
                    (/ divisor 10)
                    (conj output digit))))))))

(defn str->int4
  "Decode an encoded int4 from a string."
  [s]
  (let [v (#?(:cljs js/parseInt :clj Long/parseLong) s)]
    (int (- v offset))))
