(ns freedomdb.frontend
  (:refer-clojure :exclude [update])
  (:require
   [clojure.data.avl :as avl]
   [clojure.set :refer [difference intersection union]]
   [clojure.string :as string]
   [farbetter.utils :as u :refer
    [#?@(:clj [inspect sym-map]) throw-far-error]]
   [freedomdb.durable-row-store :as drs]
   [freedomdb.mem-row-store :as mrs]
   [freedomdb.row-store :as rs]
   [freedomdb.schemas :refer
    [DBType ExpressionType FieldAttrsMapType FieldNameType FieldsMapType
     ModifyFieldAttrsMapType SelectQueryType SelectReturnType
     SelectOneReturnType TableNameType UpdateQueryType ValueMapType]]
   [schema.core :as s :include-macros true]
   [taoensso.timbre :as timbre
    #?(:clj :refer :cljs :refer-macros) [debugf infof warnf errorf]])
  #?(:cljs
     (:require-macros
      [farbetter.utils :refer [inspect sym-map]])))

(declare check-default check-field-name check-insert-schema
         check-type filter-rows select* throw-non-existent-table-exception
         type-map valid-order-by?)

;;;;;;;;;;;;; API ;;;;;;;;;;;;;;;;;;

(defn create-db [db-type]
  (case db-type
    :mem (mrs/make-mem-row-store)
    :durable-rs-mem-kv (drs/make-durable-row-store :mem)))

(s/defn create-table :- DBType
  [db :- DBType
   table-name :- TableNameType
   fields-map :- FieldsMapType]
  "Create a table.
    Parameters:
    - db - DB instance
    - table-name - Keyword.
    - :fields-map - Map of field names (keywords) to a map of
         field attributes. Field attributes:
           - :type - Required. One of the supported types.
           - :indexed - Optional. Boolean. Defaults to true.
           - :default - Optional. If present, this value will be used for fields
                        not specified in insert statements. If not present, all
                        fields must be specified in insert statements."
  (when (nil? db)
    (throw-far-error "db argument cannot be nil"
                     :illegal-argument :db-is-nil
                     (sym-map db table-name)))
  (when-not (keyword? table-name)
    (throw-far-error
     "Table name must be a keyword"
     :illegal-argument :bad-table-name-type
     (sym-map table-name)))
  (when-not (map? db)
    (throw-far-error
     "db argument is not valid"
     :illegal-argument :bad-db-type
     (sym-map db)))
  (when-not (map? fields-map)
    (throw-far-error
     "fields-map parameter is not a map"
     :illegal-argument :bad-fields-map-type
     (sym-map fields-map)))
  (when-not (every? keyword? (keys fields-map))
    (throw-far-error
     "Field names must be keywords"
     :illegal-argument :bad-field-name-type
     (sym-map fields-map)))
  (reduce-kv (fn [_ field field-attrs]
               (when-not (contains? field-attrs :type)
                 (throw-far-error
                  (str "No field type given for field `" field "`")
                  :illegal-argument :no-field-type
                  (sym-map field field-attrs)))
               (let [{:keys [type]} field-attrs
                     check-fn (type-map type)]
                 (when-not check-fn
                   (throw-far-error
                    (str "Bad field type `" type "` for field `" field "`")
                    :illegal-argument :bad-field-type
                    (sym-map type field field-attrs))))
               (check-default table-name field field-attrs))
             nil fields-map)
  (let [default-opts {:indexed true}
        xf-fm (fn [acc k v]
                (let [{:keys [type]} v
                      new-v (merge default-opts v)]
                  (assoc acc k new-v)))
        fields-map (reduce-kv xf-fm {} fields-map)]
    (rs/create-table db table-name fields-map)))

(s/defn drop-table :- DBType
  [db :- DBType
   table-name :- TableNameType]
  "Drops a table."
  (when (nil? db)
    (throw-far-error "db argument cannot be nil"
                     :illegal-argument :db-is-nil
                     (sym-map db table-name)))
  (rs/drop-table db table-name))

(s/defn add-field :- DBType
  [db :- DBType
   table-name :- TableNameType
   field-name :- FieldNameType
   field-attrs-map :- FieldAttrsMapType]
  "Add a field to a table.
     Parameters:
     - db - DB instance
     - table-name - Keyword.
     - field-name - Keyword.
     - field-attrs-map - Attributes:
       - :type - Required. One of the supported types.
       - :indexed - Optional. Boolean. Defaults to true.
       - :default - Optional. If present, this value will be used for fields
                    not specified in insert statements. If not present, all
                    fields must be specified in insert statements."
  (when (nil? db)
    (throw-far-error "db argument cannot be nil"
                     :illegal-argument :db-is-nil
                     (sym-map db table-name)))
  (let [table-metadata (rs/get-metadata db table-name)]
    (when-not table-metadata
      (throw-non-existent-table-exception table-name)))
  (rs/add-field db table-name field-name field-attrs-map))

(s/defn remove-field :- DBType
  [db :- DBType
   table-name :- TableNameType
   field-name :- FieldNameType]
  "Remove a field from a table.
     Parameters:
     - db - DB instance
     - table-name - Keyword
     - field - Keyword."
  (when (nil? db)
    (throw-far-error "db argument cannot be nil"
                     :illegal-argument :db-is-nil
                     (sym-map db table-name)))
  (let [table-metadata (rs/get-metadata db table-name)]
    (when-not table-metadata
      (throw-non-existent-table-exception table-name))
    (check-field-name table-metadata field-name "remove-field")
    (rs/remove-field db table-name field-name)))

(s/defn modify-field :- DBType
  [db :- DBType
   table-name :- TableNameType
   field-name :- FieldNameType
   field-attrs-map :- FieldAttrsMapType]
  "Modify the options of a field in a table.
     Parameters:
     - db - DB instance
     - table-name - Keyword.
     - field-name - Keyword.
     - field-attrs-map - Attributes:
       - :type - One of the supported types.
       - :indexed - Boolean
       - :default - If present, this value will be used for fields
                    not specified in insert statements. If not present, all
                    fields must be specified in insert statements."
  (when (nil? db)
    (throw-far-error "db argument cannot be nil"
                     :illegal-argument :db-is-nil
                     (sym-map db table-name)))
  (let [table-metadata (rs/get-metadata db table-name)]
    (when-not table-metadata
      (throw-non-existent-table-exception table-name))
    (check-field-name table-metadata field-name "modify-field")
    (check-default table-name field-name field-attrs-map
                   (get-in table-metadata [:type-map field-name]))
    (rs/modify-field db table-name field-name field-attrs-map)))

(s/defn get-table-names-set :- #{TableNameType}
  "Returns the names of tables in the db as a set."
  [db :- DBType]
  (when (nil? db)
    (throw-far-error "db argument cannot be nil"
                     :illegal-argument :db-is-nil
                     (sym-map db)))
  (rs/get-table-names-set db))

(s/defn insert :- DBType
  [db :- DBType
   table-name :- TableNameType
   val-map :- ValueMapType]
  (when (nil? db)
    (throw-far-error "db argument cannot be nil"
                     :illegal-argument :db-is-nil
                     (sym-map db table-name)))
  (let [table-metadata (rs/get-metadata db table-name)
        _ (when-not table-metadata
            (throw-non-existent-table-exception table-name))
        {:keys [defaults indexed-fields type-map]} table-metadata
        val-map (cond->> val-map
                  defaults (merge defaults))]
    (check-insert-schema table-name val-map type-map)
    (rs/put-row db table-name val-map type-map indexed-fields)))

(s/defn update :- DBType
  "Update a table.
    Parameters:
    - db - DB instance
    - table-name - Keyword
    - update-query - Map with these keys:
      - :set - Required. Map of field names to field values.
      - :where - Optional. Vector of expressions as in the `select` statement.
                 If not present, all rows will be updated."
  [db :- DBType
   table-name :- TableNameType
   update-query :- UpdateQueryType]
  (when (nil? db)
    (throw-far-error "db argument cannot be nil"
                     :illegal-argument :db-is-nil
                     (sym-map db table-name)))
  (let [table-metadata (rs/get-metadata db table-name)
        _ (when-not table-metadata
            (throw-non-existent-table-exception table-name))
        {:keys [all-fields type-map indexed-fields]} table-metadata
        {:keys [where]} update-query
        val-map (:set update-query)
        non-existent-set-fields (-> (keys val-map)
                                    set
                                    (difference all-fields)
                                    seq)
        row-ids (if where
                  (filter-rows db where table-metadata)
                  :__all__)]
    (when non-existent-set-fields
      (throw-far-error
       (str "Non-existent field `" (first non-existent-set-fields)
            "` in :set clause")
       :illegal-argument :non-existent-field
       {:table-name table-name
        :non-existent-field (first non-existent-set-fields)
        :query update-query}))
    (reduce #(check-type (val-map %2) (type-map %2))
            nil (keys val-map))
    (cond
      (nil? row-ids) db
      (= :__all__ row-ids) (reduce (fn [db [row-id old-val-map]]
                                     (rs/update-row
                                      db table-name row-id old-val-map
                                      val-map type-map indexed-fields))
                                   db (rs/get-all-rows db table-name
                                                       :row-ids-and-value-maps))
      :else (reduce (fn [db row-id]
                      (let [old-val-map (rs/get-row db table-name row-id)]
                        (rs/update-row db table-name row-id old-val-map
                                       val-map type-map indexed-fields)))
                    db row-ids))))

(s/defn select :- SelectReturnType
  [db :- DBType
   query :- SelectQueryType]
  (when (nil? db)
    (throw-far-error "db argument cannot be nil"
                     :illegal-argument :db-is-nil
                     (sym-map db query)))
  (select* db query))

(s/defn select-one :- SelectOneReturnType
  "Returns one result / row or nil if no rows were found.
  Verifies that there is only one result returned
  from the query. If the result has a single field, that field is returned.
  Otherwise, a vector of fields is returned. Throws an exception if more
  than one result is found."
  [db :- DBType
   query :- SelectQueryType]
  (when (nil? db)
    (throw-far-error "db argument cannot be nil"
                     :illegal-argument :db-is-nil
                     (sym-map db query)))
  (let [result (select db query)]
    (case (count result)
      0 nil
      1 (first result)
      (throw-far-error "select-one query returned more than one result"
                       :execution-error
                       :select-one-returned-more-than-one-result
                       (sym-map result)))))

(s/defn delete :- DBType
  "Deletes rows from the database.
   Parameters:
   - db - A DB instance. Required.
   - table-name - Keyword. Required.
   - where - A where clause to specify which rows to delete. Defaults to all
        rows. Optional."
  ([db :- DBType
    table-name :- TableNameType]
   (delete db table-name nil))
  ([db :- DBType
    table-name :- TableNameType
    where :- (s/maybe ExpressionType)]
   (when (nil? db)
     (throw-far-error "db argument cannot be nil"
                      :illegal-argument :db-is-nil
                      (sym-map db table-name where)))
   (let [table-metadata (rs/get-metadata db table-name)
         _ (when-not table-metadata
             (throw-non-existent-table-exception table-name))
         {:keys [type-map]} table-metadata
         row-ids (if where
                   (filter-rows db where table-metadata)
                   :__all__)]
     (cond
       (nil? row-ids) db
       (= :__all__ row-ids) (rs/delete-all-rows db table-name type-map)
       :else (reduce (fn [db row-id]
                       (rs/delete-row db table-name row-id type-map))
                     db row-ids)))))

;;;;;;;;;;; Exceptions ;;;;;;;;;;;;

(defn- throw-non-existent-field-exception [table-name field]
  (throw-far-error (str "Field `" field "` does not exist in table `"
                        table-name "`.")
                   :illegal-argument :non-existent-field
                   (sym-map field table-name)))

(defn- throw-non-existent-table-exception [table-name]
  (throw-far-error
   (str "Table '" table-name "' does not exist")
   :illegal-argument :non-existent-table
   (sym-map table-name)))

;;;;;;;;;;;;;;;;;;;;;;;;;;; Helper fns ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defn- check-default
  ([table-name field field-attrs type]
   (when (contains? field-attrs :default)
     (when (nil? (:default field-attrs))
       (throw-far-error
        (str "Default value for field `" field "` cannot be nil")
        :illegal-argument :default-cannot-be-nil
        (sym-map table-name field field-attrs)))
     (check-type (:default field-attrs) type)))
  ([table-name field field-attrs]
   (check-default table-name field field-attrs (:type field-attrs))))

(defn- check-int4 [value]
  (when-not (integer? value)
    (throw-far-error
     (str "Value `" value "` is not an integer")
     :illegal-argument :value-not-integer
     {:value value}))
  (when (or (> value 2147483647)
            (< value -2147483648))
    (throw-far-error
     (str "Absolute value of`" value "` is too large for :int4")
     :illegal-argument :abs-value-too-large-for-int4
     (sym-map value))))

(defn- check-int8 [value]
  (when-not (u/long? value)
    (throw-far-error
     (str "Value `" value "` is not an :int8")
     :illegal-argument :value-not-long
     {:value value})))

;; TODO: Check UTF-8 octets length, not just char count
(defn- check-str1000 [value]
  (when-not (string? value)
    (throw-far-error
     (str "Value `" value "` is not a string")
     :illegal-argument :value-not-string
     {:value value}))
  (when (> (count value) 1000)
    (throw-far-error
     "String is too large for :str1000 type. Only 1000 bytes are allowed."
     :illegal-argument :string-too-large
     {:value value})))

(defn- check-kw [value]
  (when-not (keyword? value)
    (throw-far-error
     (str "Value `" value "` is not a keyword")
     :illegal-argument :value-not-keyword
     {:value value}))
  ;; Make sure name length is <= 1000
  (check-str1000 (name value)))

(defn- check-bool [value]
  (when-not (contains? #{true false} value)
    (throw-far-error
     (str "Value `" value "` is not a boolean")
     :illegal-argument :value-not-boolean
     {:value value})))

(def type-map
  {:int4 check-int4
   :int8 check-int8
   :str1000 check-str1000
   :kw check-kw
   :bool check-bool
   :any identity})

(defn- check-type [v t]
  (let [check-fn (type-map t)]
    (check-fn v)))

(defn- check-insert-schema [table-name val-map type-map]
  (if (= (set (keys type-map)) (set (keys val-map)))
    (doseq [key (keys type-map)]
      (check-type (val-map key) (type-map key)))
    (let [all-fields (set (keys type-map))
          query-fields (set (keys val-map))
          missing-fields (vec (difference all-fields
                                          query-fields))
          extra-fields (vec (difference query-fields all-fields))]
      (when (seq missing-fields)
        (throw-far-error
         (str "Non-default field(s) missing. Table: " table-name
              ". Missing fields: " missing-fields)
         :illegal-argument :missing-fields
         (sym-map missing-fields table-name)))
      (when (seq extra-fields)
        (throw-far-error
         (str "Extra fields present. Table: " table-name
              ". Extra fields: " extra-fields)
         :illegal-argument :extra-fields
         (sym-map extra-fields table-name))))))

(defn- check-field-name [table-metadata field-name fn-name]
  (let [{:keys [all-fields table-name]} table-metadata]
    (when-not (contains? all-fields field-name)
      (throw-far-error
       (str "Non-existent field `" field-name "` in " fn-name " call.")
       :illegal-argument :non-existent-field
       {:table-name table-name
        :non-existent-field field-name}))))

(defn- exec-and [db expr table-metadata]
  (apply intersection expr))

(defn- exec-or [db expr table-metadata]
  (apply union expr))

(defn- exec-not [db row-ids table-metadata]
  (let [{:keys [table-name]} table-metadata]
    (reduce (fn [acc row-id]
              (if-not (contains? row-ids row-id)
                (conj acc row-id)
                acc))
            #{} (rs/get-all-rows db table-name :row-ids-only))))

(defn- exec-binop [db expr table-metadata]
  (let [[op field v & rest] expr
        _ (when-not (zero? (count rest))
            (throw-far-error "Too many args in clause"
                             :illegal-argument :too-many-args-in-clause
                             (sym-map expr)))
        {:keys [table-name all-fields indexed-fields type-map]} table-metadata
        v-type (type-map field)]
    (when-not (contains? all-fields field)
      (throw-non-existent-field-exception table-name field))
    (when-not (contains? indexed-fields field)
      (throw-far-error
       (str "Non-indexed field `" field "` used in where clause")
       :execution-error :non-indexed-field-used-in-where-clause
       (sym-map table-name field op)))
    (if (= := op)
      (rs/get-row-ids-eq db table-name field v v-type)
      (if (= :any v-type)
        (throw-far-error (str "Cannot use inequality operations on a column "
                              "of type :any.")
                         :execution-error :cant-use-ineq-ops-on-any
                         (sym-map field v-type table-name))
        (rs/get-row-ids-ineq- db table-name field v v-type op)))))

(defn- exec-scan-fn [db expr table-metadata]
  (let [[op scan-fn & scan-fields] expr
        {:keys [all-fields table-name]} table-metadata]
    (let [unknown-fields (difference (set scan-fields) all-fields)]
      (when (seq unknown-fields)
        (throw-non-existent-field-exception
         table-name (first unknown-fields))))
    (let [call-scan-fn (if-not scan-fields
                         scan-fn
                         (fn [val-map]
                           (->> scan-fields
                                (mapv val-map)
                                (apply scan-fn))))]
      (reduce (fn [acc [row-id val-map]]
                (if (call-scan-fn val-map)
                  (conj acc row-id)
                  acc))
              #{} (rs/get-all-rows db table-name :row-ids-and-value-maps)))))

(defn throw-too-many-args-in-not [args]
  (throw-far-error "More than one arg in :not clause."
                   :illegal-argument :too-many-args-in-not
                   (sym-map args)))

(def binops #{:= :< :> :<= :>=})

(defn- make-evaluator [db table-metadata]
  (fn eval-expr [expr]
    (let [[op & rest] expr
          eval-subs #(for [sub-expr rest]
                       (eval-expr sub-expr))
          [f arg] (cond
                    (contains? binops op) [exec-binop expr]
                    (= :or op) [exec-or (eval-subs)]
                    (= :and op) [exec-and (eval-subs)]
                    (= :not op) [exec-not (if-not (= 1 (count rest))
                                            (throw-too-many-args-in-not rest)
                                            (eval-expr (first rest)))]
                    (= :scan-fn op) [exec-scan-fn expr]
                    :else (throw-far-error
                           (str "Unsupported operator `" op "`")
                           :illegal-argument :unsupported-operator
                           (sym-map op expr)))]
      (f db arg table-metadata))))

(defn filter-rows [db where table-metadata]
  (let [evaluator (make-evaluator db table-metadata)]
    (evaluator where)))

(defn- group-maps [maps fields aggregate-clause]
  (let [fields (if (keyword? fields)
                 [fields]
                 fields)]
    (if (and aggregate-clause
             (seq fields))
      (group-by #(select-keys % fields) maps)
      {{} maps})))

(defn- get-agg-fn [agg-fn-name]
  (case agg-fn-name
    :max max
    :min min
    :sum +
    :count (fn [& nums] (count nums))))

(defn- throw-illegal-star-exception [agg-fn-name]
  (throw-far-error "Star (:*) can only be used with :count"
                   :illegal-argument :star-not-with-count
                   {:arg agg-fn-name}))

(defn aggregate-groups [grouped-maps aggregate-clause]
  (if-not aggregate-clause
    (grouped-maps {})
    (let [[agg-fn-name field-name as result-name] aggregate-clause
          _ (when (and (= :* field-name)
                       (not= :count agg-fn-name))
              (throw-illegal-star-exception agg-fn-name))
          agg-fn (get-agg-fn agg-fn-name)]
      (for [[k v] grouped-maps]
        (let [extracted-vals (map #(% field-name) v)
              filtered-vals (if (= :* field-name)
                              extracted-vals
                              (filter identity extracted-vals))
              aggregate-result (when (seq filtered-vals)
                                 (apply agg-fn filtered-vals))]
          (assoc k result-name aggregate-result))))))

(defn- make-comp-fn [order-pairs]
  (fn [& args]
    (loop [[field dir & rest] order-pairs]
      (when-not dir
        (throw-far-error
         "Illegal order-by clause. Must have an even number of args."
         :illegal-argument :illegal-order-by-clause
         (sym-map field dir order-pairs)))
      (let [comparison (apply compare (map #(% field) args))
            direction-modifier (case dir
                                 :asc 1
                                 :desc -1
                                 (throw-far-error
                                  (str "Illegal order-by direction `" dir
                                       "`")
                                  :illegal-argument
                                  :illegal-order-by-direction
                                  (sym-map field dir order-pairs)))]
        (if (zero? comparison)
          (if rest
            (recur rest)
            0)
          (* direction-modifier comparison))))))

(defn- sort-maps [maps order-by-clause]
  (if (seq order-by-clause)
    (sort (make-comp-fn order-by-clause) maps)
    maps))

(defn- emit-results [maps emit-fields]
  (seq
   (cond
     (keyword? emit-fields) (mapv emit-fields maps)
     (nil? emit-fields) maps
     (empty? emit-fields) maps
     :else (for [m maps]
             (mapv m emit-fields)))))

(defn- make-order-fields-set [order-by]
  (loop [[field dir & rest] order-by
         fields #{}]
    (if field
      (recur rest (conj fields field))
      fields)))

(defn- get-project-fields
  [regular-fields aggregate-field aggregate-name order-fields join-fields]
  (if-not (seq regular-fields)
    :__all__
    (cond-> regular-fields
      aggregate-field (conj aggregate-field)
      (seq order-fields) (union (disj order-fields aggregate-name))
      (seq join-fields) (union join-fields))))

(defn- transform-query [db query]
  (let [{:keys [tables where fields order-by aggregate]} query
        tables (if (keyword? tables)
                 [tables]
                 tables)
        fields-seq (if (keyword? fields)
                     [fields]
                     fields)
        fields (when-not (= [] fields)
                 fields)
        ;; Turn any empty vectors into nil
        [where order-by aggregate] (mapv seq [where order-by aggregate])
        _ (when (> 1 (count tables))
            (throw-far-error "More than one table in query"
                             :illegal-argument :more-than-one-table
                             (sym-map query tables)))
        table-name (first tables)
        table-metadata (rs/get-metadata db table-name)
        _ (when-not table-metadata
            (throw-non-existent-table-exception table-name))
        {:keys [all-fields]} table-metadata
        [_ aggregate-field _ aggregate-name] aggregate
        regular-fields (disj (set fields-seq) aggregate-name)
        non-existent-regular-fields (difference regular-fields all-fields)
        _ (when (seq non-existent-regular-fields)
            (throw-non-existent-field-exception
             table-name (first non-existent-regular-fields)))
        order-fields (make-order-fields-set order-by)
        non-existent-order-fields (difference order-fields
                                              (conj all-fields aggregate-name))
        project-fields (get-project-fields regular-fields aggregate-field
                                           aggregate-name
                                           order-fields nil)]
    (when (seq non-existent-order-fields)
      (throw-non-existent-field-exception
       table-name (first non-existent-order-fields)))
    (when (and aggregate-field
               (not (or (contains? all-fields aggregate-field)
                        (= :* aggregate-field))))
      (throw-non-existent-field-exception table-name aggregate-field))
    (sym-map table-name where order-by aggregate project-fields fields
             table-metadata)))

(defn- get-filtered-maps [db table-name project-fields
                          where-clause table-metadata]
  (let [row-ids (if where-clause
                  (filter-rows db where-clause table-metadata)
                  :__all__)]
    (let [transform-map (if (= :__all__ project-fields)
                          identity
                          #(select-keys % project-fields))]
      (cond
        (nil? row-ids) []
        (= :__all__ row-ids) (map transform-map
                                  (rs/get-all-rows db table-name
                                                   :value-maps-only))
        :else (map #(transform-map (rs/get-row db table-name %))
                   row-ids)))))

(defn- find-table [field metadata-map]
  (let [match (fn [acc table metadata]
                (let [{:keys [all-fields]} metadata]
                  (if (contains? all-fields field)
                    (conj acc table)
                    acc)))
        matching-tables (reduce-kv match [] metadata-map)]
    (if (<= (count matching-tables) 1)
      (first matching-tables)
      (throw-far-error
       (str "Amiguous field. Field `" field "` appears in multiple tables: "
            matching-tables)
       :illegal-argument :ambiguous-field
       (sym-map field matching-tables)))))

(defn- split-qualified-field [field]
  (let [[table-name field-name & rest] (string/split (name field) #"\.")]
    (when rest
      (throw-far-error
       "Field has more than one dot in name"
       :illegal-argument :field-has-more-than-one-dot
       (sym-map field)))
    [(keyword table-name) (keyword field-name)]))

(defn- qualify-field [input-field metadata-map]
  (when input-field
    (let [[table field] (split-qualified-field input-field)]
      (if-not field
        (let [table (find-table input-field metadata-map)]
          (when table
            (keyword (str (name table) "." (name input-field)))))
        (let [metadata (metadata-map table)]
          (when-not metadata
            (throw-non-existent-table-exception table))
          (when-not (contains? (:all-fields metadata) field)
            (throw-non-existent-field-exception table field))
          input-field)))))

(defn- qualify-fields [fields metadata-map]
  (mapv #(qualify-field % metadata-map) fields))

(defn- make-metadata-map [db tables]
  (reduce (fn [acc table]
            (let [metadata (rs/get-metadata db table)]
              (if-not metadata
                (throw-non-existent-table-exception table))
              (assoc acc table metadata)))
          {} tables))

(defn- get-all-fields [tables metadata-map]
  (reduce (fn [acc table]
            (let [fields (get-in metadata-map [table :all-fields])]
              (reduce #(conj %1 (-> (str (name table) "." (name %2))
                                    (keyword)))
                      acc fields)))
          [] tables))

(defn make-field-adder [metadata-map]
  (fn [acc q-field]
    (let [[table field] (split-qualified-field q-field)
          field-type (get-in metadata-map [table :type-map field])
          indexed-fields (get-in metadata-map [table :indexed-fields])
          indexed? (contains? indexed-fields field)
          default (get-in metadata-map [table :defaults field])]
      (assoc acc q-field (cond-> {:type field-type}
                           (not indexed?) (assoc :indexed false)
                           default (assoc :default default))))))

(defn- populate-join-table [new-db join-table join-pairs project-fields old-db]
  (let [qualify (fn [table field]
                  (keyword (str (name table) "." (name field))))
        add-field (fn [acc q-field]
                    (let [[table field] (split-qualified-field q-field)]
                      (clojure.core/update acc table #(if %
                                                        (conj % field)
                                                        [field]))))
        project-fields-map (reduce add-field {} project-fields)
        join (fn [db [q-field-a q-field-b]]
               (let [[table-a field-a] (split-qualified-field q-field-a)
                     [table-b field-b] (split-qualified-field q-field-b)
                     table-a-size (rs/get-row-count old-db table-a)
                     table-b-size (rs/get-row-count old-db table-b)
                     [lg-table sm-table
                      lg-field sm-field] (if (>= table-a-size table-b-size)
                                           [table-a table-b field-a field-b]
                                           [table-b table-a field-b field-a])
                     sm-q-field (qualify sm-table sm-field)
                     lg-fields (project-fields-map lg-table)
                     q-lg-fields (mapv #(qualify lg-table %) lg-fields)
                     sm-fields (project-fields-map sm-table)
                     q-sm-fields (mapv #(qualify sm-table %) sm-fields)
                     sm-vals (select old-db {:tables [sm-table]
                                             :fields sm-fields})
                     sm-rows (mapv #(zipmap q-sm-fields %) sm-vals)
                     g (fn [db sm-row]
                         (let [
                               sm-join-val (sm-q-field sm-row)
                               lg-vals (select
                                        old-db
                                        {:tables [lg-table]
                                         :fields lg-fields
                                         :where [:= lg-field sm-join-val]})]
                           (if-not lg-vals
                             db
                             (let [lg-rows (map #(zipmap q-lg-fields %)
                                                lg-vals)
                                   insert-row (fn [db lg-row]
                                                (insert db join-table
                                                        (merge lg-row sm-row)))]
                               (reduce insert-row db lg-rows)))))]
                 (reduce g db sm-rows)))]
    (reduce join new-db join-pairs)))

(defn- make-join-db-and-query [db query]
  (let [{:keys [tables join fields order-by aggregate]} query
        tables (if (keyword? tables)
                 [tables]
                 tables)
        _ (when (odd? (count join))
            (throw-far-error
             "The number of fields in the the join clause must be even."
             :illegal-argument :odd-number-of-join-fields
             (sym-map query join)))
        metadata-map (make-metadata-map db tables)
        [_ aggregate-field _ aggregate-name] aggregate
        aggregate-field (qualify-field aggregate-field metadata-map)
        fields-seq (if (keyword? fields)
                     [fields]
                     fields)
        regular-fields (-> (set fields-seq)
                           (disj aggregate-name)
                           (qualify-fields metadata-map)
                           (set))
        order-fields (make-order-fields-set order-by)
        join-q-fields (qualify-fields join metadata-map)
        project-fields (get-project-fields regular-fields aggregate-field
                                           aggregate-name
                                           order-fields (set join-q-fields))
        project-fields (if (= :__all__ project-fields)
                         (get-all-fields tables metadata-map)
                         project-fields)
        join-pairs (partition 2 join-q-fields)
        join-table (-> (mapv name tables)
                       (conj "join")
                       (#(string/join "__" %))
                       (keyword))
        field-map (reduce (make-field-adder metadata-map) {} project-fields)
        new-db (-> (create-db :mem)
                   (create-table join-table field-map)
                   (populate-join-table join-table join-pairs project-fields
                                        db))
        qualify-no-agg (fn [field]
                         (if (= field aggregate-name)
                           field
                           (qualify-field field metadata-map)))
        new-fields (if (keyword? fields)
                     (qualify-no-agg fields)
                     (mapv qualify-no-agg fields))
        new-query (-> query
                      (assoc :tables [join-table])
                      (assoc :fields new-fields)
                      (dissoc :join))]
    [new-db new-query]))

;; Should we allow implicit tables clause (using fully qualified names
;; in :join clause? Not right now. May be a bad idea in general.
(defn- select* [db query]
  (let [{:keys [join tables]} query
        tables (if (keyword? tables)
                 [tables]
                 tables)
        [db query] (case (count tables)
                     0 (throw-far-error
                        "Missing tables clause in select."
                        :illegal-argument :missing-tables-clause
                        (sym-map query tables))
                     1 (if-not (seq join)
                         [db query]
                         (throw-far-error
                          (str "Select queries with join clauses must have "
                               "more than one table listed in the :tables "
                               "clause.")
                          :illegal-argument :join-with-only-one-table
                          (sym-map query join tables)))
                     (if (seq join)
                       (make-join-db-and-query db query)
                       (throw-far-error
                        "Missing join clause for multi-table query."
                        :illegal-argument :missing-join-clause
                        (sym-map query tables join))))
        tquery (transform-query db query)
        {:keys [where order-by fields table-metadata]} tquery
        {:keys [fields aggregate project-fields]} tquery
        {:keys [table-name]} table-metadata
        maps (get-filtered-maps db table-name project-fields
                                where table-metadata)
        grouped-maps (group-maps maps fields aggregate)
        agg-maps (aggregate-groups grouped-maps aggregate)
        sorted-maps (sort-maps agg-maps order-by)]
    (emit-results sorted-maps fields)))

;; TODO: Write input validators and good error messages (Replace
;; schema for user-facing validation)
