(ns farbetter.freedomdb.mem-row-store
  (:require
   [clojure.data.avl :as avl]
   [clojure.set :refer [union]]
   [farbetter.freedomdb.row-store :as rs :refer [RowStore]]
   [farbetter.freedomdb.schemas :refer
    [FieldName FieldsMap RowId ValueMap Value]]
   [farbetter.utils :as u
    :refer [throw-far-error #?@(:clj [inspect sym-map])]]
   [schema.core :as s :include-macros true])
  #?(:cljs
     (:require-macros
      [farbetter.utils :refer [inspect sym-map]])))

(def VRI {FieldName {Value #{RowId}}})
(def RVI {RowId ValueMap})
(def Table
  {(s/required-key :fields-map) FieldsMap
   (s/required-key :vri) VRI
   (s/required-key :rvi) RVI})

;;;;;;;;; Helper fns ;;;;;;;;;

(defn remove-val-from-vri [vri field-name v row-id]
  (let [old-row-ids (get-in vri [field-name v])
        updated-old-row-ids (disj old-row-ids row-id)]
    (if (= #{} updated-old-row-ids)
      (update vri field-name dissoc v)
      (assoc-in vri [field-name v] updated-old-row-ids))))

(defn add-val-to-vri [vri field-name v row-id]
  (update-in vri [field-name v] #(if %
                                   (conj % row-id)
                                   #{row-id})))

(defn add-to-vri [vri row-id val-map indexed-fields]
  (reduce-kv (fn [acc field-name v]
               (if-not (contains? indexed-fields field-name)
                 acc
                 (add-val-to-vri acc field-name v row-id)))
             vri val-map))

(defn remove-from-vri [vri row-id val-map]
  (reduce-kv (fn remove-field [acc field-name v]
               (remove-val-from-vri acc field-name v row-id))
             vri val-map))

(defn- update-vri [vri row-id old-val-map set-val-map indexed-fields]
  (let [f (fn [acc field-name new-val]
            (if-not (contains? indexed-fields field-name)
              acc
              (let [old-val (old-val-map field-name)]
                (-> acc
                    (remove-val-from-vri field-name old-val row-id)
                    (add-val-to-vri field-name new-val row-id)))))]
    (reduce-kv f vri set-val-map)))

(defn fields-map->metadata [table-name fields-map]
  (let [initial-metadata {:all-fields #{} :indexed-fields #{}
                          :type-map {} :defaults {}}
        xfm (fn [m field attr-map]
              (let [{:keys [indexed type default]} attr-map
                    {:keys [all-fields indexed-fields type-map defaults]} m]
                (cond-> (assoc m :all-fields (conj all-fields field))
                  indexed (assoc :indexed-fields (conj indexed-fields field))
                  (some? default) (assoc :defaults
                                         (assoc defaults field default))
                  true (assoc :type-map (assoc type-map field type)))))]
    (-> (reduce-kv xfm initial-metadata fields-map)
        (assoc :table-name table-name))))

(defn fields-map->vri [fields-map]
  (reduce-kv (fn [acc field-name field-attrs]
               (assoc acc field-name
                      (if (= :any (:type field-attrs))
                        {} ;; :any doesn't support sorting
                        (avl/sorted-map))))
             {} fields-map))
;;;;;;;;; Record ;;;;;;;;;

(defrecord MemRowStore [tables next-row-id]
  RowStore
  (create-table- [this table-name fields-map]
    (let [vri (fields-map->vri fields-map)
          rvi {}
          metadata (fields-map->metadata table-name fields-map)]
      (assoc-in this [:tables table-name]
                (sym-map fields-map vri rvi metadata))))

  (drop-table- [this table-name]
    (update this :tables dissoc table-name))

  (get-metadata- [this table-name]
    (get-in tables [table-name :metadata]))

  (get-table-names-set- [this]
    (set (keys tables)))

  ;; TODO: Implement these 3 fns
  (add-field- [this table-name field-name attrs]
    (throw-far-error "add-field is not supported by MemRowStore"
                     :execution-error :unsupported-method {}))

  (drop-field- [this table-name field-name]
    (throw-far-error "remove-field is not supported by MemRowStore"
                     :execution-error :unsupported-method {}))

  (modify-field- [this table-name field-name attrs]
    (throw-far-error "modify-field is not supported by MemRowStore"
                     :execution-error :unsupported-method {}))

  (get-row-count- [this table-name]
    (count (get-in tables [table-name :rvi])))

  (get-row- [this table-name row-id]
    (get-in tables [table-name :rvi row-id]))

  (get-all-rows- [this table-name format]
    (case format
      :row-ids-and-value-maps (seq (get-in tables [table-name :rvi]))
      :row-ids-only (keys (get-in tables [table-name :rvi]))
      :value-maps-only (vals (get-in tables [table-name :rvi]))))

  (get-row-ids-eq- [this table-name field-name v v-type]
    (get-in tables [table-name :vri field-name v]))

  (get-row-ids-ineq- [this table-name field-name v v-type op]
    (let [index (get-in tables [table-name :vri field-name])
          [subseq-fn subseq-op] (case op
                                  :< [rsubseq <]
                                  :<= [rsubseq <=]
                                  :> [subseq >]
                                  :>= [subseq >=])]
      (reduce (fn [acc [v row-ids]]
                (union acc row-ids))
              #{} (subseq-fn index subseq-op v))))

  (put-row- [this table-name val-map type-map indexed-fields]
    (let [row-id next-row-id]
      (-> this
          (update :next-row-id inc)
          (assoc-in [:tables table-name :rvi row-id] val-map)
          (update-in [:tables table-name :vri]
                     add-to-vri row-id val-map indexed-fields))))

  (update-row- [this table-name row-id old-val-map set-val-map
                type-map indexed-fields]
    (let [merged-val-map (merge old-val-map set-val-map)]
      (-> this
          (assoc-in [:tables table-name :rvi row-id] merged-val-map)
          (update-in [:tables table-name :vri]
                     update-vri row-id old-val-map
                     set-val-map indexed-fields))))

  (delete-row- [this table-name row-id type-map]
    (let [val-map (rs/get-row- this table-name row-id)]
      (-> this
          (update-in [:tables table-name :rvi]
                     dissoc row-id)
          (update-in [:tables table-name :vri]
                     remove-from-vri row-id val-map))))

  (delete-all-rows- [this table-name type-map]
    (let [vri (fields-map->vri type-map)
          rvi {}]
      (-> this
          (assoc-in [:tables table-name :vri] vri)
          (assoc-in [:tables table-name :rvi] rvi)))))

;;;;;;;;; Constructor ;;;;;;;;;

(defn make-mem-row-store []
  (->MemRowStore {} 0))
