(ns farbetter.freedomdb
  (:refer-clojure :exclude [update])
  (:require
   [farbetter.freedomdb.frontend :as fe]
   [farbetter.freedomdb.schemas :as schemas :refer
    [DB Expression FieldAttrsMap FieldName FieldsMap
     ModifyFieldAttrsMap RowStoreType SelectQuery SelectReturn SelectOneReturn
     TableName TableNameOrTableNameSeq UpdateQuery ValueMap]]
   [farbetter.freedomdb.transit :as transit]
   [farbetter.utils :as u :refer
    [#?@(:clj [inspect sym-map]) throw-far-error]]
   [schema.core :as s :include-macros true]
   [taoensso.timbre :as timbre
    #?(:clj :refer :cljs :refer-macros) [debugf infof warnf errorf]])
  #?(:cljs
     (:require-macros
      [farbetter.utils :refer [inspect sym-map]])
     :clj
     (:import [java.io ByteArrayInputStream ByteArrayOutputStream]
              [org.joda.time DateTime])))

(declare edn->transit transit->edn)

;;;;;;;;;;;;; API ;;;;;;;;;;;;;;;;;;

(s/defn create-db :- DB
  "Creates a DB with the given storage backend."
  [row-store-type :- RowStoreType]
  (fe/create-db row-store-type))

(s/defn create-table :- DB
  "Create a table.
    Parameters:
    - db - DB instance
    - table-name - Keyword.
    - :fields-map - Map of field names (keywords) to a map of
         field attributes. Field attributes:
           - :type - Required. One of the supported types.
           - :indexed - Optional. Boolean. Defaults to true.
           - :default - Optional. If present, this value will be used for fields
                        not specified in insert statements. If not present, all
                        fields must be specified in insert statements."
  [db :- DB
   table-name :- TableName
   fields-map :- FieldsMap]
  (fe/create-table db table-name fields-map))

(s/defn drop-table :- DB
  "Drops a table."
  [db :- DB
   table-name :- TableName]
  (fe/drop-table db table-name))

(s/defn add-field :- DB
  "Add a field to a table.
     Parameters:
     - db - DB instance
     - table-name - Keyword.
     - field-name - Keyword.
     - field-attrs-map - Attributes:
       - :type - Required. One of the supported types.
       - :indexed - Optional. Boolean. Defaults to true.
       - :default - Optional. If present, this value will be used for fields
                    not specified in insert statements. If not present, all
                    fields must be specified in insert statements."
  [db :- DB
   table-name :- TableName
   field-name :- FieldName
   field-attrs-map :- FieldAttrsMap]
  (fe/add-field db table-name field-name field-attrs-map))

(s/defn drop-field :- DB
  "Remove a field from a table.
     Parameters:
     - db - DB instance
     - table-name - Keyword
     - field - Keyword."
  [db :- DB
   table-name :- TableName
   field-name :- FieldName]
  (fe/drop-field db table-name field-name))

(s/defn modify-field :- DB
  "Modify the options of a field in a table.
     Parameters:
     - db - DB instance
     - table-name - Keyword.
     - field-name - Keyword.
     - field-attrs-map - Attributes:
       - :type - One of the supported types.
       - :indexed - Boolean
       - :default - If present, this value will be used for fields
                    not specified in insert statements. If not present, all
                    fields must be specified in insert statements."
  [db :- DB
   table-name :- TableName
   field-name :- FieldName
   field-attrs-map :- FieldAttrsMap]
  (fe/modify-field db table-name field-name field-attrs-map))

(s/defn get-table-names-set :- #{TableName}
  "Returns the names of tables in the db as a set."
  [db :- DB]
  (fe/get-table-names-set db))

(s/defn insert :- DB
  "Insert a row into the database."
  [db :- DB
   table-name :- TableName
   val-map :- ValueMap]
  (fe/insert db table-name val-map))

(s/defn update :- DB
  "Update a table.
    Parameters:
    - db - DB instance
    - table-name - Keyword
    - update-query - Map with these keys:
      - :set - Required. Map of field names to field values.
      - :where - Optional. Vector of expressions as in the `select` statement.
                 If not present, all rows will be updated."
  [db :- DB
   table-name :- TableName
   update-query :- UpdateQuery]
  (fe/update db table-name update-query))

(s/defn select :- SelectReturn
  "Select from the database."
  ([db :- DB
    table-name-or-table-name-seq :- TableNameOrTableNameSeq]
   (select db table-name-or-table-name-seq {}))
  ([db :- DB
    table-name-or-table-name-seq :- TableNameOrTableNameSeq
    query :- SelectQuery]
   (fe/select db table-name-or-table-name-seq query)))

(s/defn select-one :- SelectOneReturn
  "Returns one result / row or nil if no rows were found.
  Verifies that there is only one result returned
  from the query. If the result has a single field, that field is returned.
  Otherwise, a vector of fields is returned. Throws an exception if more
  than one result is found."
  ([db :- DB
    table-name-or-table-name-seq :- TableNameOrTableNameSeq]
   (select-one db table-name-or-table-name-seq {}))
  ([db :- DB
    table-name-or-table-name-seq :- TableNameOrTableNameSeq
    query :- SelectQuery]
   (fe/select-one db table-name-or-table-name-seq query)))

(s/defn delete :- DB
  "Deletes rows from the database.
   Parameters:
   - db - A DB instance. Required.
   - table-name - Keyword. Required.
   - where - A where clause to specify which rows to delete. Defaults to all
        rows. Optional."
  ([db :- DB
    table-name :- TableName]
   (delete db table-name nil))
  ([db :- DB
    table-name :- TableName
    arg :- (s/maybe schemas/DeleteArg)]
   (fe/delete db table-name arg)))

(s/defn db->transit :- s/Str
  ([db :- DB]
   (db->transit db {}))
  ([db :- DB
    addl-handlers :- schemas/TransitWriteHandlersMap]
   (transit/edn->transit db addl-handlers)))

(s/defn transit->db :- DB
  ([s :- s/Str]
   (transit->db s {}))
  ([s :- s/Str
    addl-handlers :- schemas/TransitReadHandlersMap]
   (transit/transit->edn s addl-handlers)))
