(ns farbetter.mu.websocket
  (:require
   [farbetter.utils :as u :refer
    [throw-far-error #?@(:clj [go-safe inspect sym-map])]]
   #?(:clj [gniazdo.core :as ws])
   [schema.core :as s :include-macros true]
   [taoensso.timbre :as timbre
    #?(:clj :refer :cljs :refer-macros) [debugf errorf infof]])
  #?(:cljs
     (:require-macros
      [farbetter.utils :refer [go-safe inspect sym-map]])
     :clj
     (:import [java.util Arrays])))

#?(:clj
   (defn make-websocket-clj
     [url on-connect on-disconnect on-error on-rcv]
     (let [on-bin (fn [bytes offset length]
                    (on-rcv (Arrays/copyOfRange bytes offset
                                                (+ offset length))))
           socket (ws/connect url
                              :on-connect (fn [session]
                                            (on-connect))
                              :on-close (fn [status reason]
                                          (on-disconnect reason))
                              :on-error on-error
                              :on-receive on-rcv
                              :on-binary on-bin)
           closer #(ws/close socket)
           sender (fn [data]
                    ;; Send-msg mutates binary data, so we make a
                    ;; copy if data is binary
                    (if (string? data)
                      (ws/send-msg socket data)
                      (ws/send-msg socket (Arrays/copyOf data
                                                         (count data)))))]
       (sym-map closer sender))))

#?(:cljs
   (defn make-websocket-node
     [url on-connect on-disconnect on-error on-rcv]
     (let [WSC (goog.object.get (js/require "websocket") "client")
           client (WSC.)
           conn-atom (atom nil)
           msg-handler (fn [msg-obj]
                         (let [type (goog.object.get msg-obj "type")
                               data (if (= "utf8" type)
                                      (goog.object.get msg-obj "utf8Data")
                                      (-> (goog.object.get msg-obj "binaryData")
                                          (js/Int8Array.)))]
                           (on-rcv data)))
           conn-handler (fn [conn]
                          (reset! conn-atom conn)
                          (on-connect)
                          (.on conn "close" (fn [reason description]
                                              (on-disconnect description)))
                          (.on conn "message" msg-handler)
                          (.on conn "error" on-error))
           closer #(if @conn-atom
                     (.close @conn-atom)
                     (.abort client))
           sender (fn [data]
                    (if (string? data)
                      (.sendUTF @conn-atom data)
                      (.sendBytes @conn-atom (js/Buffer. data))))
           failure-handler  (fn [err]
                              (let [reason [:connect-failure err]]
                                (on-error reason)
                                (on-disconnect reason)))]
       (.on client "connectFailed" failure-handler)
       (.on client "connect" conn-handler)
       (.connect client url)
       (sym-map closer sender))))

#?(:cljs
   (defn make-websocket-browser
     ;; TODO: Test and fully implement
     [url on-connect on-disconnect on-error on-rcv]
     (let [socket (js/WebSocket. url)
           closer #(.close socket)
           sender #(.send socket %)]
       (.onopen socket on-connect)
       (.onclose socket (fn [] (on-disconnect :closed)))
       (.onerror socket on-error)
       (.onmessage socket #(on-rcv (goog.object.get % "data")))
       (set! (.-binaryType socket) "arraybuffer")
       (sym-map closer sender))))

#?(:cljs
   (defn make-websocket-ios
     ;; TODO: Test and fully implement
     [url on-connect on-disconnect on-error on-rcv]
     (let [socket (.makeWithCallbacks js/FarWebSocket url on-connect
                                      on-disconnect on-error on-rcv)
           closer #(.close socket)
           sender #(.send socket %)]
       (sym-map closer sender))))

(s/defn make-websocket :- {(s/required-key :sender) (s/=> s/Any)
                           (s/required-key :closer) (s/=> s/Any)}
  [url :- s/Str
   on-connect :- (s/=> s/Any)
   on-disconnect :- (s/=> s/Any)
   on-error :- (s/=> s/Any)
   on-rcv :- (s/=> s/Any)]
  (try
    (let [platform (u/get-platform-kw)
          factory #?(:clj make-websocket-clj
                     :cljs (case platform
                             :node make-websocket-node
                             :ios make-websocket-ios
                             :browser make-websocket-browser))]
      (factory url on-connect on-disconnect on-error on-rcv))
    (catch #?(:clj Exception :cljs :default) e
      (let [reason [:construction-failed e]]
        (on-error reason)
        (on-disconnect reason))
      nil)))
