(ns farbetter.mu.gw-router
  (:require
   [#?(:clj clojure.core.async :cljs cljs.core.async) :as ca]
   [farbetter.mu.utils :as mu :refer [ConnId MsgId RequestId]]
   [farbetter.roe.schemas :as rs :refer [AvroData AvroName]]
   [farbetter.utils :as u :refer
    [throw-far-error ByteArray #?@(:clj [go-safe inspect sym-map])]]
   [freedomdb.frontend :as fdb]
   [freedomdb.schemas :refer [DBType]]
   [taoensso.timbre :as timbre
    #?(:clj :refer :cljs :refer-macros) [debugf errorf infof]]
   [schema.core :as s :include-macros true])
  #?(:cljs
     (:require-macros
      [farbetter.utils :as u :refer [go-safe inspect sym-map]])))

(def policy-load-interval-ms 3000)

;; call-through and weighed are based on code from
;; https://github.com/clojure/data.generators/blob/master/src/main/clojure/clojure/data/generators.clj
;; by Stuart Halloway

(defn- call-through
  "Recursively call x until it doesn't return a function."
  [x]
  (if (fn? x)
    (recur (x))
    x))

(defn weighted
  "Given a map of generators and weights, return a value from one of
   the generators, selecting generator based on weights."
  [m]
  (let [weights (reductions + (vals m))
        total (last weights)
        choices (map vector (keys m) weights)]
    (let [choice (rand total)]
      (loop [[[c w] & more] choices]
        (when w
          (if (< choice w)
            (call-through c)
            (recur more)))))))

(defn choose-rule [rules]
  (let [m (reduce (fn [acc rule]
                    (assoc acc rule (:weight rule)))
                  {} rules)]
    (weighted m)))

(defn evaluate-rules [rules user-id rpc-rq-msg]
  (let [user-rules (filter (fn [rule]
                             (let [{:keys [users]} rule
                                   users-set (set users)]
                               (contains? users-set user-id))) rules)]
    (if (seq user-rules)
      (choose-rule user-rules)
      (let [star-rules (filter #(= (:users %) []) rules)]
        (when (seq star-rules)
          (choose-rule star-rules))))))

;; TODO: Rewrite this using a join
(defn get-rule-for-rpc [db rpc-rq-msg user-id]
  (let [{:keys [service-api-version]} rpc-rq-msg
        {:keys [service-name major-version]} service-api-version
        version-pairs (fdb/select
                       db {:tables [:sis]
                           :fields [:minor-version :micro-version]
                           :where [:and
                                   [:= :service-name service-name]
                                   [:= :major-version major-version]]})
        vp-set (set version-pairs)
        rules (fdb/select db {:tables [:traffic-policy-rules]
                              :where [:and
                                      [:= :service-name service-name]
                                      [:= :major-version major-version]]})
        rules (filter (fn [rule]
                        (let [{:keys [minor-version micro-version]} rule
                              version-pair [minor-version micro-version]]
                          (contains? vp-set version-pair)))
                      rules)]
    (evaluate-rules rules user-id rpc-rq-msg)))

(s/defn route-rpc-rq :- (s/maybe ConnId)
  [db :- DBType
   rpc-rq-msg :- AvroData]
  (let [{:keys [request-id user-id]} rpc-rq-msg
        rule (get-rule-for-rpc db rpc-rq-msg user-id)]
    (when rule
      (let [{:keys [service-name major-version
                    minor-version micro-version]} rule
            si-conn-ids (fdb/select
                         db
                         {:tables [:sis]
                          :fields :si-conn-id
                          :where [:and
                                  [:= :service-name service-name]
                                  [:= :major-version major-version]
                                  [:= :minor-version minor-version]
                                  [:= :micro-version micro-version]]})]
        (rand-nth si-conn-ids)))))

(s/defn request-id->client-conn-id :- (s/maybe ConnId)
  [db :- DBType
   request-id :- RequestId]
  (let [client-conn-id (fdb/select-one db {:tables [:gw-rpcs]
                                           :fields :client-conn-id
                                           :where [:= :request-id request-id]})]
    client-conn-id))
