(ns farbetter.mu.client
  (:require
   [#?(:clj clojure.core.async :cljs cljs.core.async) :as ca
    :refer [#? (:clj go)]]
   [farbetter.freedomdb :as fdb]
   [farbetter.freedomdb.schemas :refer [DB]]
   [farbetter.mu.msgs :as msgs :refer [APISpec]]
   [farbetter.mu.msg-xf :as mxf]
   [farbetter.mu.proc :as proc]
   [farbetter.mu.state :as state]
   [farbetter.mu.transport :as tp]
   [farbetter.mu.utils :as mu :refer
    [bad-login command-block Command CommandOrCommandBlock ConnId
     RequestId UserId]]
   [farbetter.mu.websocket :as websocket]
   [farbetter.pete :as pete]
   [farbetter.roe :as roe]
   [farbetter.roe.schemas :refer [AvroData]]
   [farbetter.utils :as u :refer
    [throw-far-error #?@(:clj [inspect sym-map])]]
   [schema.core :as s :include-macros true]
   [taoensso.timbre :as timbre
    #?(:clj :refer :cljs :refer-macros) [debugf errorf infof tracef]])
  #?(:cljs
     (:require-macros
      [cljs.core.async.macros :refer [go]]
      [farbetter.utils :as u :refer [inspect sym-map]])))

(def rpc-check-interval-ms 500)

(def UserIdOrBadLogin (s/if keyword?
                        (s/eq :bad-login)
                        UserId))

(declare call-rpc* check-for-failed-login get-gw-conn-id wait-for-login*)

(defprotocol IRPCClient
  (call-rpc
    [this service-name fn-name arg]
    [this service-name fn-name arg opts]
    "Valid opts:
     - :timeout-ms - Number of milliseconds before timeout.
     - :on-success - Fn of one arg (ret) to be called on when the RPC succeeds.
     - :on-failure - Fn of one arg (reason) to be called on when the
                     RPC fails.
     *** Note that if no :on-success callback option is given, call-rpc will
     return a success/failure channel."))

(defprotocol IClient
  (collect-garbage [this]
    "Force garbage collection. Useful for testing.")
  (disconnect [this]
    "Disconnect the client from the network. Useful for testing")
  (stop [this]
    "Stop processing loop. Useful for testing.")
  (wait-for-login
    [this]
    [this timeout-ms]
    "Wait until the client is logged in to exactly one gateway or times out.
     Returns a success/failure channel."))

(defrecord Client [db-atom active?-atom command-chan rcv-chan apis repeater]
  IRPCClient
  (call-rpc [this service-name fn-name arg]
    (call-rpc this service-name fn-name arg {}))
  (call-rpc [this service-name fn-name arg opts]
    (call-rpc* apis service-name fn-name arg command-chan rcv-chan
               db-atom :cl #(wait-for-login this %) #(get-gw-conn-id @db-atom)
               opts))

  IClient
  (collect-garbage [this]
    (ca/put! command-chan [[:collect-garbage]]))

  (disconnect [this]
    (let [conn-ids (state/get-conn-ids @db-atom)]
      (doseq [conn-id conn-ids]
        (ca/put! command-chan [[:close-and-delete-conn conn-id]]))))

  (stop [this]
    (tracef ":cl stop called")
    (proc/close-conns! @db-atom)
    (pete/stop repeater)
    (reset! active?-atom false)
    (ca/close! rcv-chan)
    (ca/close! command-chan))

  (wait-for-login [this]
    (wait-for-login this mu/default-login-wait-ms))
  (wait-for-login [this timeout-ms]
    (wait-for-login* db-atom timeout-ms)))

;;;;;;;;;;;;;;;;;;;; Helper fns ;;;;;;;;;;;;;;;;;;;;

(defn get-gw-conn-id [db]
  (let [li-conn-ids (state/get-logged-in-conn-ids db)]
    (cond
      (nil? li-conn-ids) nil
      (= 1 (count li-conn-ids) 1) (first li-conn-ids)
      :else (throw-far-error "More than one logged-in connection!!"
                             :execution-error :multiple-logged-in-conns
                             (sym-map li-conn-ids)))))

(s/defn get-user-id :- (s/maybe UserIdOrBadLogin)
  [db :- DB]
  (fdb/select-one db :user-id {:fields :user-id}))

(s/defn set-user-id :- DB
  [db :- DB
   user-id :- UserIdOrBadLogin]
  (if (get-user-id db)
    db
    (fdb/insert db :user-id (sym-map user-id))))

(defn- check-for-failed-login [db]
  (when (= bad-login (get-user-id db))
    (throw-far-error "Login failed due to bad credentials."
                     :execution-error :login-failed-bad-credentials {})))

(defn make-connect-to-gw [gw-urls-factory rcv-chan command-chan db-atom
                          conn-factory]
  (s/fn client-connect-to-gw :- nil
    [db :- DB]
    (tracef "Entering client-connect-to-gw.")
    (u/go-sf
     (let [gw-urls (u/call-sf! gw-urls-factory)]
       (doseq [url gw-urls]
         (tp/connect-to-gw url rcv-chan command-chan db-atom
                           conn-factory :cl))))
    nil))

(s/defn update-client-conns :- (s/maybe [CommandOrCommandBlock])
  [db :- DB]
  (when (not= bad-login (get-user-id db))
    (when-not (get-gw-conn-id db)
      (let [nli-conn-ids (state/get-non-logged-in-conn-ids db)]
        (if (pos? (count nli-conn-ids))
          (when (every? #(state/timed-out? db %) nli-conn-ids)
            (let [cad-commands (mapv (fn [conn-id]
                                       [:close-and-delete-conn conn-id])
                                     nli-conn-ids)
                  block (-> (apply command-block cad-commands)
                            (conj [:connect-to-gw]))]
              [block]))
          [[:connect-to-gw]])))))

(s/defn handle-client-login-rs :- [CommandOrCommandBlock]
  [db :- DB
   conn-id :- ConnId
   msg :- AvroData]
  (let [{:keys [was-successful user-id]} msg]
    (if was-successful
      (if (state/get-logged-in-conn-ids db)
        (do
          (tracef (str ":cl login to %s was successful, but was not the "
                       "first :gw connection. Closing connection.")
                  conn-id)
          [[:close-and-delete-conn conn-id]])
        (do
          (tracef ":cl login to %s was successful. user-id: %s"
                  conn-id (u/int-map->hex-str user-id))
          [(command-block
            [:set-user-id user-id]
            [:set-gw-conn-if-not-set conn-id]
            [:close-and-delete-if-not-gw-conn conn-id])]))
      (do
        (tracef (str ":cl login to %s failed due to bad credentials. "
                     "Closing connection.") conn-id)
        [(command-block
          [:set-user-id bad-login]
          [:close-and-delete-conn conn-id])]))))

(s/defn close-and-delete-if-not-gw-conn :- (s/maybe [CommandOrCommandBlock])
  [db :- DB
   conn-id :- ConnId]
  (when-not (= conn-id (get-gw-conn-id db))
    [[:close-and-delete-conn conn-id]]))

(s/defn set-gw-conn-if-not-set :- DB
  [db :- DB
   conn-id :- ConnId]
  (if (state/get-logged-in-conn-ids db)
    ;; If there is already a logged-in conn, we don't update the db
    db
    (state/set-conn-state db conn-id :logged-in)))

(s/defn handle-rpc-rs-success :- (s/maybe [CommandOrCommandBlock])
  [db :- DB
   conn-id :- ConnId
   msg :- AvroData]
  (tracef "Got rpc-rs-success. Request-id %s" (u/int-map->hex-str
                                               (:request-id msg)))
  (let [{:keys [request-id return-schema-fp encoded-return-value]} msg
        w-schema (state/fingerprint->schema db return-schema-fp)
        row (fdb/select-one db :client-rpcs
                            {:where (sym-map request-id)})]
    (when row
      (let [{:keys [on-success service-name fn-name]} row
            return-schema-name (state/make-rpc-schema-name
                                service-name fn-name :return)]
        (if w-schema
          (let [return-schema-fp (state/name->fingerprint db return-schema-name)
                return-schema (state/fingerprint->schema db return-schema-fp)
                ret (roe/avro-byte-array->edn w-schema return-schema
                                              encoded-return-value)]
            (on-success ret)
            [[:delete-client-rpc request-id]])
          (let [enc-msg (roe/edn->avro-byte-array
                         msgs/rpc-rs-success-schema msg)
                msg-fp (roe/edn-schema->fingerprint msgs/rpc-rs-success-schema)]
            (mxf/get-missing-schema-commands
             db conn-id return-schema-fp msg-fp enc-msg)))))))

(s/defn handle-rpc-rs-failure :- (s/maybe [CommandOrCommandBlock])
  [db :- DB
   conn-id :- ConnId
   msg :- AvroData]
  (let [{:keys [request-id reason]} msg
        request-id-str (u/int-map->hex-str request-id)
        _ (tracef "Client got rq-failed msg. request-id: %s reason: %s"
                  request-id-str reason)
        on-failure (fdb/select-one db :client-rpcs
                                   {:fields :on-failure
                                    :where (sym-map request-id)})]
    (if on-failure
      (on-failure {:request-id request-id-str :reason reason})
      (tracef "No on-failure handler found."))
    [[:delete-client-rpc request-id]]))

(s/defn delete-client-rpc :- DB
  [db :- DB
   request-id :- RequestId]
  (fdb/delete db :client-rpcs [:= :request-id request-id]))

(s/defn add-client-rpc :- DB
  [db :- DB
   request-id :- RequestId
   on-success :- (s/=> s/Any)
   on-failure :- (s/=> s/Any)
   service-name :- s/Str
   fn-name :- s/Str
   timeout-ms :- s/Num]
  (let [expiry-time-ms (+ timeout-ms (u/get-current-time-ms))]
    (fdb/insert db :client-rpcs
                (sym-map request-id on-success on-failure
                         service-name fn-name expiry-time-ms))))

(s/defn modify-db :- DB
  [db :- DB]
  (-> db
      (fdb/create-table
       :client-rpcs {:request-id {:type :any :indexed true}
                     :on-success {:type :any :indexed false}
                     :on-failure {:type :any :indexed false}
                     :service-name {:type :str1000 :indexed false}
                     :fn-name {:type :str1000 :indexed false}
                     :expiry-time-ms {:type :num
                                      :indexed true}})
      (fdb/create-table
       :user-id {:user-id {:type :any :indexed false}})))

(s/defn gc-client-rpcs :- (s/maybe [CommandOrCommandBlock])
  [db :- DB]
  (let [now (u/get-current-time-ms)
        expired-rpcs (fdb/select db :client-rpcs
                                 {:where [:<= :expiry-time-ms now]})]
    (for [{:keys [request-id on-failure service-name fn-name]} expired-rpcs]
      (let [request-id-str (u/int-map->hex-str request-id)
            reason "RPC timed out"]
        ;; on-failure is a side effect
        (on-failure (sym-map reason request-id-str service-name fn-name))
        [:delete-client-rpc request-id]))))

(defn- check-apis [apis]
  (let [service-names (map :service-name apis)
        name-freqs (frequencies service-names)]
    (doseq [[name count] name-freqs]
      (when (> count 1)
        (throw-far-error (str "Duplicate service name `" name "`.")
                         :illegal-argument :duplicate-service-name
                         (sym-map name apis name-freqs service-names))))))

(defn- apis->service-name->major-version* [apis]
  (reduce (fn [acc api]
            (let [{:keys [service-name major-version]} api]
              (assoc acc service-name major-version)))
          {} apis))

(def apis->service-name->major-version
  (memoize apis->service-name->major-version*))

(defn wait-for-login* [db-atom timeout-ms]
  (u/go-sf
   (let [expiry-ms (+ (u/get-current-time-ms) timeout-ms)]
     (loop []
       (cond
         (= bad-login (get-user-id @db-atom))
         (let [reason ":cl wait-for-login* failed due to bad credentials."]
           (throw-far-error reason :execution-error
                            :login-failed-bad-credentials {}))

         (>= (u/get-current-time-ms) expiry-ms)
         (let [reason ":cl wait-for-login* timed out."]
           (throw-far-error reason :execution-error
                            :timed-out-waiting-for-client-login
                            (sym-map timeout-ms)))

         (nil? (get-gw-conn-id @db-atom))
         (do
           (ca/<! (ca/timeout 10))
           (recur))

         :else
         true)))))

(defn call-rpc* [apis service-name fn-name arg command-chan rcv-chan
                 db-atom proc-type wait-for-login get-gw-conn-id opts]
  (check-for-failed-login @db-atom)
  (let [result-chan (ca/chan 1)
        {:keys [timeout-ms on-success on-failure]
         :or {timeout-ms mu/default-rpc-timeout-ms
              on-success #(ca/put! result-chan [:success %])
              on-failure #(ca/put! result-chan [:failure %])}} opts
        min-timeout (* 2 rpc-check-interval-ms)]
    (when-not (fn? on-success)
      (throw-far-error "on-success is not a fn."
                       :illegal-argument :on-success-is-not-a-fn
                       (sym-map service-name fn-name on-success on-failure)))
    (when-not (fn? on-failure)
      (throw-far-error "on-failure is not a fn."
                       :illegal-argument :on-failure-is-not-a-fn
                       (sym-map service-name fn-name on-success on-failure)))
    (when-not (string? service-name)
      (throw-far-error "service-name is not a string."
                       :illegal-argument :service-name-is-not-a-string
                       (sym-map service-name fn-name on-success on-failure)))
    (when-not (string? fn-name)
      (throw-far-error "fn-name is not a string."
                       :illegal-argument :fn-name-is-not-a-string
                       (sym-map service-name fn-name on-success on-failure)))
    (when-not (number? timeout-ms)
      (throw-far-error "timeout-ms is not a number."
                       :illegal-argument :timeout-ms-is-not-a-number
                       (sym-map service-name fn-name on-success on-failure)))
    (when (< timeout-ms min-timeout)
      (throw-far-error (str "timeout-ms cannot be shorter than "
                            min-timeout "ms.")
                       :illegal-argument :timeout-ms-too-short
                       (sym-map min-timeout timeout-ms
                                service-name fn-name)))
    (u/go-sf
     (tracef "Entering call-rpc. fn-name: %s timeout-ms: %s"
             fn-name timeout-ms)
     (try
       (let [request-id (-> (u/make-v1-uuid)
                            (u/uuid->int-map))
             service-name->major-version (apis->service-name->major-version
                                          apis)
             major-version (service-name->major-version service-name)
             _ (when-not major-version
                 (throw-far-error
                  (str "Attempt to call unregistered API: " service-name)
                  :illegal-argument :unregistered-api
                  (sym-map service-name fn-name arg proc-type opts)))
             service-api-version (sym-map service-name major-version)
             arg-schema-name (state/make-rpc-schema-name
                              service-name fn-name :arg)
             arg-schema-fp (state/name->fingerprint @db-atom arg-schema-name)
             arg-schema (state/fingerprint->schema @db-atom arg-schema-fp)
             encoded-arg (roe/edn->avro-byte-array arg-schema arg)
             _ (tracef "Waiting for / checking login...")
             [status reason] (ca/<! (wait-for-login timeout-ms))]
         (if (= :success status)
           (let [_ (tracef "%s logged in, starting RPC..." proc-type)
                 user-id (get-user-id @db-atom)
                 gw-conn-id (get-gw-conn-id)
                 _ (when-not gw-conn-id
                     (throw-far-error
                      "gw-conn-id is nil after login."
                      :execution-error :gw-conn-id-is-nil
                      (sym-map gw-conn-id service-name fn-name on-success
                               on-failure proc-type)))
                 msg (sym-map service-api-version user-id request-id fn-name
                              arg-schema-fp encoded-arg timeout-ms)]
             (debugf "%s sending %s RPC %s to :gw %s" proc-type
                     fn-name (u/int-map->hex-str request-id) gw-conn-id)
             (let [block (command-block
                          [:add-client-rpc request-id on-success
                           on-failure service-name fn-name
                           timeout-ms]
                          [:send-msg gw-conn-id :rpc-rq msg])]
               (ca/put! command-chan [block])))
           (do
             (tracef "%s call-rpc timed out waiting for login" proc-type)
             (on-failure reason))))
       (catch #?(:clj Exception :cljs :default) e
         (on-failure (u/get-exception-msg-and-stacktrace e)))))
    (when-not (:on-success opts)
      result-chan)))

;;;;;;;;;;;;;;;;;;;; Constructor ;;;;;;;;;;;;;;;;;;;;

(s/defn make-client :- (s/protocol IClient)
  "Create a mu client.
   Parameters:
   - apis - A sequence of APIs that this client will be able to talk to.
       Each API should conform to the APISpec schema. Required.
   - Username - The user's username or email address as a string. Required.
   - password - The user's password as a string. Required.
   - opts - An optional map of options. Valid options:
        - :gw-urls-factory - Fn of no arguments. Should return a success/failure
            channel. If successful, the second argument should be a sequence of
            valid gateway URLs (as strings).
        - :conn-factory - A fn to construct a connection. If not
            specfied, websockets will be used. If specified, the fn should
            accept these arguments (all are required):
                - url - The URL to connect to. String.
                - on-connect - A fn of no arguments to be called on connection.
                - on-disconnect - A fn of one argument (reason) to be called
                    on disconnection.
                - on-error - A fn of one argument (reason); called on error.
                - on-rcv - A fn of one argument (data); called when data
                    is received.
            The fn should return a map with two keys:
                - sender - A fn of one argument (data) that can be called to
                    send data over the connection.
                - closer - A fn of no arguments that can be called to close the
                    connection."
  ([apis :- [APISpec]
    username :- s/Str
    password :- s/Str]
   (make-client apis username password {}))
  ([apis :- [APISpec]
    username :- s/Str
    password :- s/Str
    opts :- {(s/optional-key :gw-urls-factory) (s/=> [s/Str])
             (s/optional-key :conn-factory) (s/=> s/Any)}]
   (check-apis apis)
   (let [{:keys [client-apis gw-urls-factory conn-factory]
          :or {gw-urls-factory #?(:clj mu/get-gw-urls
                                  :cljs #(throw-far-error
                                          "CLJS must specify a gw-urls-factory"
                                          :execution-error :no-gw-urls-factory
                                          (sym-map opts)))
               conn-factory websocket/make-client-websocket}} opts
         db-atom (atom (state/make-db apis modify-db))
         active?-atom (atom true)
         command-chan (ca/chan mu/chan-buf-size)
         rcv-chan (ca/chan mu/chan-buf-size)
         send-login (fn [db conn-id]
                      (let [msg (sym-map username password)]
                        [[:send-msg conn-id :client-login-rq msg]]))
         connect-to-gw (make-connect-to-gw gw-urls-factory rcv-chan command-chan
                                           db-atom conn-factory)
         addl-side-effect-op->f (sym-map close-and-delete-if-not-gw-conn
                                         connect-to-gw
                                         gc-client-rpcs
                                         handle-client-login-rs
                                         handle-rpc-rs-success
                                         handle-rpc-rs-failure
                                         send-login
                                         update-client-conns)
         addl-state-op-f (sym-map add-client-rpc delete-client-rpc
                                  set-gw-conn-if-not-set
                                  set-user-id)
         addl-msg->op {:rpc-rs-success :handle-rpc-rs-success
                       :rpc-rs-failure :handle-rpc-rs-failure
                       :client-login-rs :handle-client-login-rs}
         send-update-client-conns #(ca/put! command-chan
                                            [[:update-client-conns]])
         repeater (pete/make-repeater)]
     (proc/start-processor active?-atom db-atom repeater command-chan rcv-chan
                           addl-side-effect-op->f addl-state-op-f
                           addl-msg->op :cl)
     ;; Manually send update-client-conns the first time so we don't have
     ;; to wait for pete to send it
     (send-update-client-conns)
     (pete/add-fn! repeater :update-client-conns send-update-client-conns
                   mu/max-loop-wait-ms)
     (pete/add-fn! repeater :gc-client-rpcs
                   #(ca/put! command-chan [[:gc-client-rpcs]])
                   rpc-check-interval-ms)
     (map->Client (sym-map db-atom active?-atom command-chan rcv-chan
                           apis repeater)))))
