(ns farbetter.mu.gateway
  (:require
   [#?(:clj clojure.core.async :cljs cljs.core.async) :as ca]
   [farbetter.freedomdb :as fdb]
   [farbetter.freedomdb.schemas :refer [DB]]
   [farbetter.mu.gw-router :as router]
   [farbetter.mu.msgs :as msgs]
   [farbetter.mu.msg-xf :as mxf]
   [farbetter.mu.proc :as proc]
   [farbetter.mu.state :as state]
   [farbetter.mu.transport :as transport]
   [farbetter.mu.utils :as mu :refer
    [command-block Command CommandOrCommandBlock ConnId MsgId RequestId]]
   [farbetter.pete :as pete]
   [farbetter.roe :as roe]
   [farbetter.roe.schemas :as rs :refer [AvroData AvroName]]
   [farbetter.utils :as u :refer
    [throw-far-error ByteArray #?@(:clj [inspect sym-map])]]
   [schema.core :as s :include-macros true]
   [taoensso.timbre :as timbre
    #?(:clj :refer :cljs :refer-macros) [debugf errorf infof warnf]])
  #?(:cljs
     (:require-macros
      [farbetter.utils :as u :refer [inspect sym-map]])))

(declare make-on-connect)

(def rcv-chan-buf-size 1000)
(def rules-refresh-interval-ms (* 1000 2))
(def max-rpc-wait-ms (* 1000 20))

(defprotocol IGateway
  (serve [this])
  (stop [this]))

(defrecord Gateway [active?-atom db-atom repeater serve-fn stop-fn-atom
                    rcv-chan command-chan]
  IGateway
  (serve [this]
    (let [client-ws-handler (make-on-connect :cl rcv-chan command-chan db-atom)
          si-ws-handler (make-on-connect :si rcv-chan command-chan db-atom)]
      (reset! stop-fn-atom (serve-fn client-ws-handler si-ws-handler))))

  (stop [this]
    (debugf ":gw stop called")
    (@stop-fn-atom)
    (proc/close-conns! @db-atom)
    (reset! active?-atom false)
    (pete/stop repeater)
    (ca/close! rcv-chan)
    (ca/close! command-chan)))

;;;;;;;;;;;;;;;;;;;; Helper fns ;;;;;;;;;;;;;;;;;;;;

(defn make-on-connect [conn-type rcv-chan command-chan db-atom]
  (fn on-connect [conn-id sender closer]
    (when (nil? conn-type)
      (throw-far-error "conn-type is nil." :illegal-argument :conn-type-is-nil
                       (sym-map conn-id conn-type sender closer)))
    (when (nil? conn-id)
      (throw-far-error "conn-id is nil." :illegal-argument :conn-id-is-nil
                       (sym-map conn-id conn-type sender closer)))
    (when (nil? sender)
      (throw-far-error "sender is nil." :illegal-argument :sender-is-nil
                       (sym-map conn-id conn-type sender closer)))
    (when (nil? closer)
      (throw-far-error "closer is nil." :illegal-argument :closer-is-nil
                       (sym-map conn-id conn-type sender closer)))
    (let [on-disconnect (fn [reason]
                          (let [commands (cond-> [[:close-and-delete-conn
                                                   conn-id]]
                                           (= :si conn-type)
                                           (conj [:delete-si conn-id]))]
                            (debugf "GW on-disconnect. conn-id %s Reason: %s"
                                    conn-id reason)
                            (ca/put! command-chan commands)))
          on-error #(do (debugf "Error: %s" %)
                        (on-disconnect %))
          on-rcv (transport/make-on-rcv conn-id rcv-chan db-atom)]
      (debugf "In gw on-connect. conn-id: %s" conn-id)
      (ca/put! command-chan [(command-block
                              [:add-conn conn-id sender closer conn-type]
                              [:set-conn-state conn-id :connected])])
      (sym-map on-rcv on-error on-disconnect))))

(s/defn handle-rpc-rq :- (s/maybe [CommandOrCommandBlock])
  [db :- DB
   client-conn-id :- ConnId
   msg :- AvroData]
  (let [{:keys [request-id arg-schema-fp timeout-ms
                service-api-version fn-name]} msg]
    (if (state/fingerprint->schema db arg-schema-fp)
      (let [expiry-time-ms (+ timeout-ms (u/get-current-time-ms))
            [status v] (router/route-rpc-rq db msg)
            si-conn-id v]
        (if (= :success status)
          (do
            (debugf "Routing RPC %s (%s) to SI %s. Timeout-ms: %s"
                    (u/int-map->hex-str request-id) fn-name
                    si-conn-id timeout-ms)
            [(command-block [:insert-gw-rpc
                             (sym-map client-conn-id si-conn-id fn-name
                                      request-id expiry-time-ms)]
                            [:send-msg si-conn-id :rpc-rq msg])])
          (let [reason v
                rq-failed-msg (sym-map request-id reason)]
            (warnf (str reason))
            [[:send-msg client-conn-id :rpc-rs-failure rq-failed-msg]])))
      (let [enc-msg (roe/edn->avro-byte-array msgs/rpc-rq-schema msg)
            msg-fp (roe/edn-schema->fingerprint msgs/rpc-rq-schema)]
        (mxf/get-missing-schema-commands
         db client-conn-id arg-schema-fp msg-fp enc-msg)))))

(s/defn handle-rpc-rs-success :- (s/maybe [CommandOrCommandBlock])
  [db :- DB
   si-conn-id :- ConnId
   msg :- AvroData]
  (let [{:keys [request-id return-schema-fp]} msg]
    (if (state/fingerprint->schema db return-schema-fp)
      (let [client-conn-id (router/request-id->client-conn-id db request-id)
            delete-rpc-command [:delete-gw-rpc request-id]]
        (if client-conn-id
          [(command-block delete-rpc-command
                          [:send-msg client-conn-id :rpc-rs-success msg])]
          [delete-rpc-command]))
      (let [enc-msg (roe/edn->avro-byte-array msgs/rpc-rs-success-schema msg)
            msg-fp (roe/edn-schema->fingerprint msgs/rpc-rs-success-schema)]
        (mxf/get-missing-schema-commands
         db si-conn-id return-schema-fp msg-fp enc-msg)))))

(s/defn handle-rpc-rs-failure
  [db :- DB
   si-conn-id :- ConnId
   msg :- AvroData]
  (let [{:keys [request-id]} msg
        client-conn-id (router/request-id->client-conn-id db request-id)
        delete-rpc-command [:delete-gw-rpc request-id]]
    (if client-conn-id
      [(command-block delete-rpc-command
                      [:send-msg client-conn-id :rpc-rs-failure msg])]
      [delete-rpc-command])))

(defn make-si-login-handler [authenticator]
  (s/fn handle-si-login :- (s/maybe [CommandOrCommandBlock])
    [db :- DB
     conn-id :- ConnId
     msg :- AvroData]
    (let [{:keys [username password service-instance-version]} msg
          user-id (authenticator username password)
          was-successful (boolean user-id)
          send-msg-command [:send-msg conn-id :service-instance-login-rs
                            (sym-map was-successful user-id)]]
      (debugf "In si-login-handler. was-successful: %s" was-successful)

      (if was-successful
        [(command-block [:insert-si (assoc service-instance-version
                                           :si-conn-id conn-id)]
                        [:set-conn-state conn-id :logged-in]
                        send-msg-command)]
        [(command-block send-msg-command
                        [:close-and-delete-conn conn-id])]))))

(defn make-client-login-handler [authenticator]
  (s/fn handle-client-login :- (s/maybe [CommandOrCommandBlock])
    [db :- DB
     conn-id :- ConnId
     msg :- AvroData]
    (let [{:keys [username password]} msg
          user-id (authenticator username password)
          was-successful (boolean user-id)
          send-msg-cmd [:send-msg conn-id :client-login-rs
                        (sym-map was-successful user-id)]
          cmds (if was-successful
                 [(command-block [:set-conn-state conn-id :logged-in]
                                 send-msg-cmd)]
                 [(command-block send-msg-cmd
                                 [:close-and-delete-conn conn-id])])]
      (debugf "In :gw handle-client-login. commands: %s" cmds)
      cmds)))

(s/defn delete-gw-rpc :- DB
  [db :- DB
   request-id :- RequestId]
  (fdb/delete db :gw-rpcs {:where (sym-map request-id)}))

(s/defn delete-si :- DB
  [db :- DB
   conn-id :- ConnId]
  (fdb/delete db :sis {:where [:= :si-conn-id conn-id]}))

(defn modify-db [db]
  [db :- DB]
  (-> db
      (fdb/create-table :gw-rpcs {:client-conn-id {:type :any :indexed true}
                                  :si-conn-id {:type :any :indexed true}
                                  :request-id {:type :any :indexed true}
                                  :fn-name {:type :str1000 :indexed false}
                                  :expiry-time-ms {:type :num :indexed true}})
      (fdb/create-table :sis {:si-conn-id {:type :any}
                              :service-name {:type :str1000}
                              :major-version {:type :int4}
                              :minor-version {:type :int4}
                              :micro-version {:type :int4}})
      (fdb/create-table :traffic-policy-rules
                        {:service-name {:type :str1000}
                         :major-version {:type :int4}
                         :minor-version {:type :int4}
                         :micro-version {:type :int4}
                         :users {:type :any}
                         :weight {:type :int4}})))

(s/defn insert-si :- DB
  [db :- DB
   row :- {:si-conn-id ConnId
           :service-name s/Str
           :major-version s/Num
           :minor-version s/Num
           :micro-version s/Num}]
  (fdb/insert db :sis row))

(defn get-gw-rpc-info [db request-id]
  (fdb/select-one db :gw-rpcs {:where (sym-map request-id)}))

(s/defn insert-gw-rpc :- DB
  [db :- DB
   row :- {:client-conn-id ConnId
           :si-conn-id ConnId
           :request-id RequestId
           :fn-name s/Str
           :expiry-time-ms s/Num}]
  (fdb/insert db :gw-rpcs row))

(s/defn gc-gw-rpc :- (s/maybe [CommandOrCommandBlock])
  [db :- DB]
  (let [now (u/get-current-time-ms)
        expired-rpcs (fdb/select db :gw-rpcs
                                 {:where [:<= :expiry-time-ms now]})
        gc-rpc (fn [acc rpc]
                 (let [{:keys [client-conn-id request-id fn-name]} rpc
                       type "execution-error"
                       subtype "rpc-timeout"
                       error-map nil
                       exception-msg (str "RPC timed out. GW did not get "
                                          "response from SI.")
                       stacktrace nil
                       request-id-str (u/int-map->hex-str request-id)
                       reason (sym-map type subtype error-map
                                       exception-msg stacktrace request-id-str
                                       fn-name)
                       msg (sym-map request-id reason)]
                   (-> acc
                       (conj (command-block [:delete-gw-rpc request-id]
                                            [:send-msg client-conn-id
                                             :rpc-rs-failure msg])))))]
    (reduce gc-rpc [] expired-rpcs)))

(defn insert-traffic-rule [db rule-row]
  (fdb/insert db :traffic-policy-rules rule-row))

(defn make-refresh-rules [rules-factory]
  (s/fn refresh-rules :- DB
    [db :- DB]
    (let [add-rows (fn [db]
                     (reduce insert-traffic-rule db (rules-factory)))]
      (-> db
          (fdb/delete :traffic-policy-rules)
          (add-rows)))))

;;;;;;;;;;;;;;;;;;;; Constructor ;;;;;;;;;;;;;;;;;;;;

(s/defn make-gateway :- (s/protocol IGateway)
  "Create a mu gateway.
   Arguments (all are required):
   - authenticator - A fn of two arguments (username and password). If the
        username and password are valid, should return a user-id as an int-map,
        otherwise, nil.
   - rules-factory - A fn of no arguments that returns a sequence of traffic
        rules.
   - serve-fn - A fn of two arguments:
        - client-ws-handler
        - si-ws-hander
        Should return a stop fn of no arguments that can be called to stop
        the server."
  [authenticator :- (s/=> s/Any)
   rules-factory :- (s/=> s/Any)
   serve-fn :- (s/=> s/Any)]
  (let [db-atom (atom (state/make-db [] modify-db))
        repeater (pete/make-repeater)
        active?-atom (atom true)
        stop-fn-atom (atom (constantly nil))
        command-chan (ca/chan mu/chan-buf-size)
        rcv-chan (ca/chan mu/chan-buf-size)
        handle-client-login (make-client-login-handler authenticator)
        handle-si-login (make-si-login-handler authenticator)
        refresh-rules (make-refresh-rules rules-factory)
        addl-state-op->f (sym-map delete-gw-rpc delete-si insert-si
                                  insert-gw-rpc refresh-rules)
        addl-side-effect-op->f (sym-map handle-client-login handle-si-login
                                        handle-rpc-rq handle-rpc-rs-success
                                        handle-rpc-rs-failure gc-gw-rpc)
        addl-msg->op {:client-login-rq :handle-client-login
                      :service-instance-login-rq :handle-si-login
                      :rpc-rq :handle-rpc-rq
                      :rpc-rs-success :handle-rpc-rs-success
                      :rpc-rs-failure :handle-rpc-rs-failure}]
    (proc/start-processor active?-atom db-atom repeater command-chan rcv-chan
                          addl-side-effect-op->f addl-state-op->f
                          addl-msg->op :gw)
    (ca/put! command-chan [[:refresh-rules]])
    (pete/add-fn! repeater :refresh-rules
                  #(ca/put! command-chan [[:refresh-rules]])
                  rules-refresh-interval-ms)
    (pete/add-fn! repeater :gc-gw-rpc
                  #(ca/put! command-chan [[:gc-gw-rpc]])
                  mu/gc-gw-rpc-interval-ms)
    (map->Gateway
     (sym-map active?-atom db-atom repeater serve-fn stop-fn-atom rcv-chan
              command-chan))))
