(ns farbetter.mu.mock-net
  (:require
   [#?(:clj clojure.core.async :cljs cljs.core.async) :as ca]
   [farbetter.utils :as u :refer
    [throw-far-error #?@(:clj [inspect sym-map]
                         :cljs [byte-array long])]]
   [schema.core :as s :include-macros true]
   [taoensso.timbre :as timbre
    #?(:clj :refer :cljs :refer-macros) [debugf errorf infof]])
  #?(:cljs
     (:require-macros
      [farbetter.utils :refer [inspect sym-map]])))


(defprotocol INetwork
  (add-server [this server-url on-connect])
  (remove-server [this server-url])
  (connect-client
    [this client-id server-url on-rcv]
    [this client-id server-url on-rcv opts]
    "Connect a client to a server via the mock network.")
  (disconnect-client [this client-id]
    "Disconnect a client from the mock network"))

(defrecord MockNet [url->server-on-connect client-id->disconnectors]
  INetwork
  (add-server [this server-url on-connect]
    (debugf "MockNet: Adding server %s" server-url)
    (swap! url->server-on-connect assoc server-url on-connect))

  (remove-server [this server-url]
    (debugf "MockNet: Removing server %s" server-url)
    (swap! url->server-on-connect dissoc server-url))

  (connect-client [this client-id server-url on-rcv]
    (connect-client this client-id server-url on-rcv {}))

  (connect-client [this client-id server-url on-rcv opts]
    (debugf "MockNet: connecting %s to %s" client-id server-url)
    (let [svr-on-connect (@url->server-on-connect server-url)
          {:keys [on-connect on-disconnect on-error connect-ms]
           :or {connect-ms 1}} opts]
      (if-not svr-on-connect
        (let [reason (str "MockNet: Could not connect to " server-url)]
          (debugf reason)
          (if on-error
            (on-error reason)
            (u/throw-far-error reason :illegal-argument
                               :could-not-connect (sym-map server-url)))
          nil)
        (let [closer #(disconnect-client this client-id)
              ret (svr-on-connect client-id on-rcv closer)
              sender #(try
                        ((:on-rcv ret) %)
                        (catch #?(:clj Exception :cljs :default) e
                          (debugf "MockNet: Got error on send: %s" e)
                          (when on-error
                            (on-error e))))]
          (swap! client-id->disconnectors assoc client-id
                 [(:on-disconnect ret) on-disconnect])
          (when on-connect
            (u/go-sf
             (ca/<! (ca/timeout connect-ms))
             (on-connect)))
          (sym-map closer sender)))))

  (disconnect-client [this client-id]
    (doseq [disconnector (@client-id->disconnectors client-id)]
      (when disconnector
        (disconnector :explicit-close)))
    (swap! client-id->disconnectors dissoc client-id)))

(defn make-mock-net []
  (let [url->server-on-connect (atom {})
        client-id->disconnectors (atom {})]
    (map->MockNet (sym-map url->server-on-connect client-id->disconnectors))))
