(ns farbetter.mu.websocket
  (:require
   [farbetter.mu.utils :as mu]
   [farbetter.utils :as u :refer
    [throw-far-error #?@(:clj [inspect sym-map])]]
   #?(:clj [gniazdo.core :as ws])
   [schema.core :as s :include-macros true]
   [taoensso.timbre :as timbre
    #?(:clj :refer :cljs :refer-macros) [debugf errorf infof]])
  #?(:cljs
     (:require-macros
      [farbetter.utils :refer [inspect sym-map throws]])
     :clj
     (:import [java.util Arrays])))

(def wait-time-ms-ios 50)
(def ttl-connect-ios 5000)

#?(:clj
   (defn make-websocket-clj
     [url on-connect on-disconnect on-error on-rcv]
     (let [on-bin (fn [bytes offset length]
                    (on-rcv (Arrays/copyOfRange
                             ^bytes bytes ^int offset
                             ^int (+ offset length))))
           socket (try
                    (ws/connect url
                                :on-connect (fn [session]
                                              (on-connect))
                                :on-close (fn [status reason]
                                            (on-disconnect reason))
                                :on-error on-error
                                :on-receive on-rcv
                                :on-binary on-bin)
                    (catch Exception e
                      (on-error e)))
           closer #(ws/close socket)
           sender (fn [data]
                    (try
                      ;; Send-msg mutates binary data, so we make a
                      ;; copy if data is binary
                      (if (string? data)
                        (ws/send-msg socket data)
                        (ws/send-msg socket (Arrays/copyOf
                                             ^bytes data ^int (count data))))
                      (catch Exception e
                        (on-error e))))]
       (sym-map closer sender))))

#?(:cljs
   ;; TODO: When there is no connection possible, this should call
   ;; on-error or on-disconnect or something..
   (defn make-websocket-node
     [url on-connect on-disconnect on-error on-rcv]
     (let [WSC (goog.object.get (js/require "websocket") "client")
           client (WSC.)
           conn-atom (atom nil)
           msg-handler (fn [msg-obj]
                         (let [type (goog.object.get msg-obj "type")
                               data (if (= "utf8" type)
                                      (goog.object.get msg-obj "utf8Data")
                                      (-> (goog.object.get msg-obj "binaryData")
                                          (js/Int8Array.)))]
                           (on-rcv data)))
           conn-handler (fn [conn]
                          (reset! conn-atom conn)
                          (on-connect)
                          (.on conn "close" (fn [reason description]
                                              (on-disconnect description)))
                          (.on conn "message" msg-handler)
                          (.on conn "error" on-error))
           closer #(if @conn-atom
                     (.close @conn-atom)
                     (.abort client))
           sender (fn [data]
                    (if (string? data)
                      (.sendUTF @conn-atom data)
                      (.sendBytes @conn-atom (js/Buffer. data))))
           failure-handler  (fn [err]
                              (let [reason [:connect-failure err]]
                                (on-error reason)
                                (on-disconnect reason)))]
       (.on client "connectFailed" failure-handler)
       (.on client "connect" conn-handler)
       (.connect client url)
       (sym-map closer sender))))

#?(:cljs
   (defn make-websocket-jsc-ios
     [url on-connect on-disconnect on-error on-rcv]
     (try
       (let [on-rcv- (fn [data]
                       (let [size (.-length data)
                             xfed (js/Int8Array. size)
                             _ (.set xfed data)]
                         (on-rcv xfed)))
             ws (.makeWithURLOnConnectOnCloseOnErrorOnReceive
                               js/FarWebSocket url on-connect on-disconnect
                               on-error on-rcv-)
             closer #(.close ws)
             sender (fn [data]
                      (.send ws (.from js/Array data)))]
         (sym-map closer sender))
       (catch js/Error e
         (on-error (str "Exception in make-websocket-ios" e))))))

#?(:cljs
   (defn testfn
     [url]
     (u/go-sf
      (make-websocket-jsc-ios url nil nil nil nil))))

#?(:cljs
  (defn make-websocket-browser
    [url on-connect on-disconnect on-error on-rcv]
    ;; TODO: Implement this
    (throw-far-error "Not imlemented"
                     :not-implemented :no-make-websocket-browser {})))

(defn throw-unknown-platform-exception []
  (throw-far-error "Unknown platform"
                   :execution-error :unknown-platform
                   {}))

(s/defn make-websocket* :- (s/maybe {(s/required-key :sender) (s/=> s/Any)
                                    (s/required-key :closer) (s/=> s/Any)})
  [endpoint-path :- s/Str
   url :- s/Str
   on-connect :- (s/=> s/Any)
   on-disconnect :- (s/=> s/Any)
   on-error :- (s/=> s/Any)
   on-rcv :- (s/=> s/Any)]
  (try
    (let [url-w-path (str url  "/" endpoint-path)
          factory #?(:clj make-websocket-clj
                     :cljs (case (u/get-platform-kw)
                             :node make-websocket-node
                             :jsc-ios make-websocket-jsc-ios
                             :browser make-websocket-browser
                             :unknown (throw-unknown-platform-exception)))]
      (debugf "Making websocket connection to %s" url-w-path)
      (factory url-w-path on-connect on-disconnect on-error on-rcv))
    (catch #?(:clj Exception :cljs :default) e
      (let [msg #?(:clj (.getMessage e) :cljs e)
            reason [:connection-failure msg]]
        (on-error reason)
        (infof "Websocket to %s failed to connect. Reason: %s" url reason))
      nil)))

(def make-client-websocket (partial make-websocket* mu/client-endpoint-path))
(def make-si-websocket (partial make-websocket* mu/si-endpoint-path))
