(ns farbetter.mu.service-instance
  (:require
   [#?(:clj clojure.core.async :cljs cljs.core.async) :as ca]
   [farbetter.mu.msgs :as msgs :refer [APISpec FnMap]]
   [farbetter.mu.msg-xf :as mxf]
   [farbetter.mu.proc :as proc]
   [farbetter.mu.state :as state]
   [farbetter.mu.transport :as tp]
   [farbetter.mu.utils :as mu :refer
    [Command CommandOrCommandBlock ConnId ConnState Fingerprint
     ProcType RequestId]]
   [farbetter.mu.websocket :as websocket]
   [farbetter.pete :as pete]
   [farbetter.roe :as roe]
   [farbetter.roe.schemas :as rs :refer [AvroData AvroSchema]]
   [farbetter.utils :as u :refer
    [throw-far-error #?@(:clj [go-safe inspect sym-map])]]
   [freedomdb.frontend :as fdb]
   [freedomdb.schemas :refer [DBType]]
   [schema.core :as s :include-macros true]
   [taoensso.timbre :as timbre
    #?(:clj :refer :cljs :refer-macros) [debugf errorf infof]])
  #?(:cljs
     (:require-macros
      [farbetter.utils :as u :refer [go-safe inspect sym-map]])))

(def gw-conn-update-interval-ms (* 1000 1))

;;;;;;;;;;;;;;;;;;;; Protocol & Record ;;;;;;;;;;

(defprotocol IServiceInstance
  (wait-for-login [this] "Returns a channel")
  (start [this] "Start serving requests.")
  (stop [this] "Stop serving requests.")
  (update-conns [this] "Update gateway connections."))

(defrecord ServiceInstance [db-atom gw-urls-factory tp-factory rcv-chan
                            command-chan access-key svc-fns api repeater
                            service-instance-version active?-atom
                            login-failed?-atom]
  IServiceInstance
  (wait-for-login [this]
    (go-safe
     (loop []
       (when @login-failed?-atom
         (throw-far-error "SI login failed due to incorrect access key."
                          :execution-error :failed-si-login {}))
       (let [[status arg]  (ca/<! (gw-urls-factory))
             _ (when-not (= :success status)
                 (throw-far-error
                  (str "gw-urls-factory call failed. reason: " arg)
                  :execution-error :gw-urls-call-failed {:reason arg}))
             gw-urls (set arg)
             logged-in-conn-ids (set (state/get-logged-in-conn-ids @db-atom))]
         (when-not (= gw-urls logged-in-conn-ids)
           (ca/<! (ca/timeout 5))
           (recur))))))

  (start [this]
    (reset! active?-atom true)
    (pete/start repeater))

  (stop [this]
    (reset! active?-atom false)
    (pete/stop repeater))

  (update-conns [this]
    (go-safe ;; This is required because we do a <! below...
     (let [[status arg] (ca/<! (gw-urls-factory))
           _ (when-not (= :success status)
               (throw-far-error
                (str "gw-urls-factory call failed. reason: " arg)
                :execution-error :gw-urls-call-failed {:reason arg}))
           gw-urls (set arg)
           current-urls (set (state/get-conn-ids @db-atom))
           added-urls (clojure.set/difference gw-urls current-urls)
           deleted-urls (clojure.set/difference current-urls gw-urls)]
       (when (or (seq added-urls)
                 (seq deleted-urls))
         (debugf "Update conns:\n  cur: %s\n  added: %s\n  deleted: %s"
                 current-urls added-urls deleted-urls))
       (doseq [url added-urls]
         (tp/connect-to-gw url rcv-chan command-chan tp-factory :si))
       (doseq [url deleted-urls]
         (ca/put! command-chan [[:close-and-delete-conn url]]))))))

;;;;;;;;;;;;;;;;;;;; Helper fns ;;;;;;;;;;;;;;;;;;;;

(defn make-handle-si-login-rs [login-failed?-atom]
  (s/fn handle-si-login-rs :- DBType
    [db :- DBType
     conn-id :- ConnId
     msg :- AvroData]
    (if (:was-successful msg)
      (state/set-conn-state db conn-id :logged-in)
      (do
        (reset! login-failed?-atom true)
        (throw-far-error "SI login failed due to incorrect access key."
                         :execution-error :failed-si-login {})))))

(s/defn execute-fn :- nil
  [db :- DBType
   conn-id :- ConnId
   request-id :- RequestId
   return-schema-fp :- Fingerprint
   fn-name :- s/Str
   f :- (s/=> s/Any)
   arg :- AvroData
   command-chan :- u/Channel]
  (go-safe
   (debugf "Entering :si execute-fn. fn-name: %s" fn-name)
   (try
     (let [ret-or-ch (f arg)
           ret (if (u/channel? ret-or-ch)
                 (ca/<! ret-or-ch)
                 ret-or-ch)
           return-schema (state/fingerprint->schema db return-schema-fp)
           encoded-return-value (roe/edn->avro-byte-array return-schema ret)
           ret-msg (sym-map request-id return-schema-fp encoded-return-value)]
       (ca/put! command-chan [[:send-msg conn-id :rpc-rs-success ret-msg]]))
     (catch #?(:clj Exception :cljs :default) e
       (let [reason (str "Exception in service-fn `" fn-name "` "
                         "request-id: " (u/int-map->hex-str request-id)
                         " Exception: " e)]
         (infof reason)
         (ca/put! command-chan [[:send-msg conn-id :rpc-rs-failure
                                 (sym-map request-id reason)]])))))
  nil)

(defn make-rpc-rq-handler [service-instance-version svc-fns command-chan]
  (s/fn handle-rpc-rq :- (s/maybe [CommandOrCommandBlock])
    [db :- DBType
     conn-id :- ConnId
     msg :- AvroData]
    (debugf "Entering :si handle-rpc-rq. msg: %s" msg)
    (let [{:keys [service-name]} service-instance-version
          {:keys [request-id fn-name arg-schema-fp encoded-arg]} msg
          arg-schema-name (state/make-rpc-schema-name service-name fn-name :arg)
          return-schema-name (state/make-rpc-schema-name
                              service-name fn-name :return)
          w-schema (state/fingerprint->schema db arg-schema-fp)]
      (if w-schema
        (let [arg-schema-fp (state/name->fingerprint db arg-schema-name)
              return-schema-fp (state/name->fingerprint db return-schema-name)
              arg-schema (state/fingerprint->schema db arg-schema-fp)
              arg (roe/avro-byte-array->edn w-schema arg-schema encoded-arg)
              f (svc-fns fn-name)]
          (execute-fn db conn-id request-id return-schema-fp fn-name f arg
                      command-chan))
        (let [enc-msg (roe/edn->avro-byte-array msgs/rpc-rq-schema msg)
              msg-fp (roe/edn-schema->fingerprint msgs/rpc-rq-schema)
              cmds (mxf/get-missing-schema-commands
                    db conn-id arg-schema-fp msg-fp enc-msg)]
          (debugf "#### In si handle-rpc-rq. msg: %s msg-bytes: commands: %s"
                  msg bytes cmds)
          cmds)))))

;;;;;;;;;;;;;;;;;;;; Constructor ;;;;;;;;;;;;;;;;;;;;

(s/defn make-service-instance :- (s/protocol IServiceInstance)
  "Create a service instance.
   Arguments:
   - api - Map describing the API of the service instance. Required keys:
        - :service-name - String. Name of the service.
        - :major-version - Integer. Major version number of the API.
        - :fns - Map of fn-name keys to fns of one arg
   - svc-fns - Map of fn-name keys to fn values.
   - minor-version - Integer. Minor version of this implementation.
   - micro-version - Integer. Micro version of this implementation.
   - access-key - String. Service instance access key. Used for GW login.
   - gw-urls-factory - Fn of no arguments. Should return a :success/:failure
        channel. If successful, the second argument should be a sequence of
        valid gateway URLs (as strings). Optional.
   - tp-factory - Factory fn to create a transport layer. Optional. Must be a
        fn of five arguments:
        - url - String. URL of the particular gateway.
        - on-rcv - Fn of one argument (data). Transport layer should call
             this fn when it receives data.
        - on-connect - Fn of no arguments. Transport layer should call this
             fn when it connects or reconnects.
        - on-disconnect - Fn of no arguments. Transport layer should call
             this fn when it disconnections.
        - on-error - Fn of one argument (error). Transport layer should call
             this fn when it encounters an error.
        Calling tp-factory must return a sender fn of one argument (data).
             This fn will be called to send data to the gateway."
  ([api :- APISpec
    svc-fns :- FnMap
    minor-version :- s/Int
    micro-version :- s/Int
    access-key :- s/Str
    gw-urls-factory :- (s/=> [s/Str])]
   (make-service-instance api svc-fns minor-version micro-version access-key
                          gw-urls-factory websocket/make-si-websocket))
  ([api :- APISpec
    svc-fns :- FnMap
    minor-version :- s/Int
    micro-version :- s/Int
    access-key :- s/Str
    gw-urls-factory :- (s/=> [s/Str])
    tp-factory :- (s/=> s/Any)]
   (let [{:keys [service-name major-version fns]} api
         api-fn-names (set (keys fns))
         svc-fn-names (set (keys svc-fns))
         missing-fns (clojure.set/difference api-fn-names svc-fn-names)
         _ (when (seq missing-fns)
             (throw-far-error (str "Missing svc-fn(s):" missing-fns)
                              :illegal-argument :missing-fns
                              (sym-map api svc-fns missing-fns)))
         command-chan (ca/chan mu/chan-buf-size)
         rcv-chan (ca/chan mu/chan-buf-size)
         repeater (pete/make-repeater)
         service-instance-version (sym-map service-name major-version
                                           minor-version micro-version)
         db-atom (atom (state/make-db [api]))
         active?-atom (atom true)
         login-failed?-atom (atom false)
         handle-si-login-rs (make-handle-si-login-rs login-failed?-atom)
         send-login (fn [db conn-id]
                      (let [msg (sym-map access-key service-instance-version)]
                        [[:send-msg conn-id :service-instance-login-rq msg]]))
         handle-rpc-rq (make-rpc-rq-handler service-instance-version svc-fns
                                            command-chan)
         addl-side-effect-op->f (sym-map handle-rpc-rq send-login)
         addl-state-op->f (sym-map handle-si-login-rs)
         addl-msg->op {:rpc-rq :handle-rpc-rq
                       :service-instance-login-rs :handle-si-login-rs}
         params (sym-map db-atom gw-urls-factory tp-factory repeater rcv-chan
                         access-key svc-fns api service-instance-version
                         active?-atom command-chan login-failed?-atom)
         si (map->ServiceInstance params)]
     (proc/start-processor active?-atom db-atom repeater command-chan
                           rcv-chan addl-side-effect-op->f addl-state-op->f
                           addl-msg->op :si)
     ;; Explicitly call update-conns the first time so we don't have
     ;; to wait for pete to schedule it
     (update-conns si)
     (pete/add-fn! repeater :update-conns #(update-conns si)
                   gw-conn-update-interval-ms)

     (start si)
     si)))
