(ns farbetter.mu.state
  (:require
   [#?(:clj clojure.core.async :cljs cljs.core.async) :as ca]
   [clojure.string :refer [join]]
   [farbetter.mu.msgs :as msgs :refer
    [APISpec FnName MajorVersion ServiceName]]
   [farbetter.mu.utils :as mu :refer
    [max-fragment-bytes ms-time-type
     APIInfo Command ConnId ConnState Fingerprint Fragment
     MsgId OpToF ProcType RequestId Timestamp]]
   [farbetter.roe :as roe]
   [farbetter.roe.schemas :as rs :refer [AvroData AvroName AvroSchema]]
   [farbetter.utils :as u :refer
    [throw-far-error ByteArray #?@(:clj [go-safe inspect sym-map])]]
   [freedomdb.frontend :as fdb]
   [freedomdb.schemas :refer [DBType]]
   [schema.core :as s :include-macros true]
   [taoensso.timbre :as timbre
    #?(:clj :refer :cljs :refer-macros) [debugf errorf infof]])
  #?(:cljs
     (:require-macros
      [farbetter.utils :as u :refer [go-safe inspect sym-map]])))

(def connect-timeout-ms (* 1000 10))
(def schema-rq-interval-ms (* 1000 5))
(def max-fragment-wait-ms mu/keep-alive-interval-ms)
(def max-schema-rq-wait-ms mu/keep-alive-interval-ms)
(def commands-per-gc 100)

(s/defn make-rpc-schema-name :- AvroName
  [service-name :- ServiceName
   fn-name :- FnName
   type :- (s/enum :arg :return)]
  (keyword (str "RPC-" (name type) "-" service-name "-" fn-name)))

(s/defn fingerprint->schema :- (s/maybe AvroSchema)
  [db :- DBType
   fingerprint :- Fingerprint]
  (fdb/select-one db {:tables [:schemas]
                      :fields :schema
                      :where [:= :fingerprint fingerprint]}))

(s/defn add-schema-name :- DBType
  [db :- DBType
   name :- s/Keyword
   fingerprint :- Fingerprint]
  (if (fdb/select db {:tables :schema-names
                      :where [:= :name name]})
    (throw-far-error (str "Schema name `" name "` already exists.")
                     :illegal-argument :duplicate-schema-name
                     (sym-map name fingerprint))
    (fdb/insert db :schema-names (sym-map name fingerprint))))

(s/defn add-edn-schema :- DBType
  [db :- DBType
   fingerprint :- Fingerprint
   schema :- AvroSchema]
  (if-let [sch (fingerprint->schema db fingerprint)]
    db
    (fdb/insert db :schemas (sym-map fingerprint schema))))

(defn- add-rpc-schemas [db api]
  (let [{:keys [service-name fn-schemas]} api
        add-rpc (fn [db [fn-name {:keys [arg-schema return-schema]}]]
                  (let [arg-fp (roe/edn-schema->fingerprint arg-schema)
                        return-fp (roe/edn-schema->fingerprint return-schema)
                        arg-name (make-rpc-schema-name
                                  service-name fn-name :arg)
                        return-name (make-rpc-schema-name
                                     service-name fn-name :return)]
                    (-> db
                        (add-edn-schema arg-fp arg-schema)
                        (add-edn-schema return-fp return-schema)
                        (add-schema-name arg-name arg-fp)
                        (add-schema-name return-name return-fp))))]
    (reduce add-rpc db fn-schemas)))

(defn- add-msg-schema [db schema]
  (let [{:keys [name]} schema ;; msgs always have a name, b/c they are :records
        fingerprint (roe/edn-schema->fingerprint schema)]
    (-> db
        (add-edn-schema fingerprint schema)
        (add-schema-name name fingerprint))))

(s/defn populate-schemas-table :- DBType
  [db :- DBType
   apis :- [APISpec]]
  (as-> db db
    (reduce add-msg-schema db mu/msg-schemas)
    (reduce add-rpc-schemas db apis)))

(s/defn make-db :- DBType
  ([apis :- [msgs/APISpec]]
   (make-db apis identity))
  ([apis :- [msgs/APISpec]
    modifier :- (s/=> DBType DBType)]
   (-> (fdb/create-db :mem)
       (fdb/create-table
        :conns {:conn-id {:type :any :indexed true}
                :sender {:type :any :indexed false}
                :closer {:type :any :indexed false}
                :type {:type :kw :indexed true}
                :state {:type :kw :indexed true}
                :start-time-ms {:type ms-time-type :indexed true}})
       (fdb/create-table
        :fragments {:msg-id {:type :any :indexed true} ;; msg-id is a map
                    :fragment-num {:type :int4 :indexed true}
                    :fragment-bytes {:type :any :indexed false}
                    :insert-time-ms {:type ms-time-type :indexed true}})
       (fdb/create-table
        :schemas {:fingerprint {:type ms-time-type :indexed true}
                  :schema {:type :any :indexed false}})
       (fdb/create-table
        :schema-names {:name {:type :kw :indexed true}
                       :fingerprint {:type ms-time-type :indexed false}})
       (fdb/create-table
        :schema-rqs {:fingerprint {:type :any :indexed true}
                     :insert-time-ms {:type ms-time-type :indexed true}})
       (fdb/create-table
        :bytes-waiting-for-schemas
        {:fingerprint {:type :any :indexed true}
         :conn-id {:type :any :indexed true}
         :bytes {:type :any :indexed false}
         :insert-time-ms {:type ms-time-type :indexed true}})
       (populate-schemas-table apis)
       (modifier))))

(s/defn get-waiting-bytes :- [[(s/one ConnId "conn-id")
                               (s/one ByteArray "bytes")]]
  [db :- DBType
   fingerprint :- Fingerprint]
  (fdb/select db {:tables [:bytes-waiting-for-schemas]
                  :fields [:conn-id :bytes]
                  :where [:= :fingerprint fingerprint]}))

(s/defn insert-bytes-waiting-for-schemas :- DBType
  [db :- DBType
   fingerprint :- Fingerprint
   conn-id :- ConnId
   bytes :- ByteArray]
  (let [insert-time-ms (u/get-current-time-ms)]
    (fdb/insert db :bytes-waiting-for-schemas
                (sym-map fingerprint conn-id bytes insert-time-ms))))

(s/defn encode-fragment :- ByteArray
  [fragment :- Fragment]
  (roe/edn->avro-byte-array msgs/fragment-schema fragment))

(defn make-fragment [msg-id fingerprint bytes fragment-num num-fragments
                     last-fragment-size]
  (let [chunk-start (* max-fragment-bytes fragment-num)
        chunk-len (if (and (= (dec num-fragments)
                              fragment-num)
                           (not (zero? last-fragment-size)))
                    last-fragment-size
                    max-fragment-bytes)
        chunk-end (+ chunk-start chunk-len)
        fragment-bytes (u/slice bytes chunk-start chunk-end)]
    (sym-map msg-id num-fragments fragment-num fingerprint fragment-bytes)))

(s/defn bytes->fragments* :- [(s/if u/byte-array? ByteArray Fragment)]
  [fingerprint :- Fingerprint
   bytes :- ByteArray
   encode? :- s/Bool]
  (let [bytes-len (count bytes)
        num-full-fragments (quot bytes-len max-fragment-bytes)
        last-fragment-size (rem bytes-len max-fragment-bytes)
        num-fragments (if (zero? last-fragment-size)
                        num-full-fragments
                        (inc num-full-fragments))
        msg-id (when (> num-fragments 1)
                 (-> (u/make-v1-uuid)
                     (u/uuid->int-map)))
        mf (fn [acc fragment-num]
             (let [fragment (make-fragment
                             msg-id fingerprint bytes fragment-num
                             num-fragments last-fragment-size)]
               (conj acc (if encode?
                           (encode-fragment fragment)
                           fragment))))]
    (reduce mf [] (range num-fragments))))

(s/defn bytes->fragments :- [Fragment]
  [fingerprint :- Fingerprint
   bytes :- ByteArray]
  (bytes->fragments* fingerprint bytes false))

(s/defn bytes->enc-fragments :- [ByteArray]
  [fingerprint :- Fingerprint
   bytes :- ByteArray]
  (bytes->fragments* fingerprint bytes true))

(s/defn add-fragment :- DBType
  [db :- DBType
   msg-id :- MsgId
   fragment-num :- s/Num
   fragment-bytes :- ByteArray]
  (let [insert-time-ms (u/get-current-time-ms)]
    (if (fdb/select db {:tables [:fragments]
                        :where [:and
                                [:= :msg-id msg-id]
                                [:= :fragment-num fragment-num]]})
      db ;; fragment exists
      (fdb/insert db :fragments (sym-map msg-id fragment-num fragment-bytes
                                         insert-time-ms)))))

(s/defn get-fragment-bytes-seq :- [ByteArray]
  [db :- DBType
   msg-id :- MsgId]
  (fdb/select db {:tables [:fragments]
                  :where [:= :msg-id msg-id]
                  :fields :fragment-bytes
                  :order-by [:fragment-num :asc]}))

(s/defn name->fingerprint :- Fingerprint
  [db :- DBType
   name :- AvroName]
  (fdb/select-one db {:tables [:schema-names]
                      :fields :fingerprint
                      :where [:= :name name]}))

(s/defn get-conn-state-and-type :- (s/maybe [(s/one ConnState "conn-state")
                                             (s/one ProcType "conn-type")])
  [db :- DBType
   conn-id :- ConnId]
  (fdb/select-one db {:tables [:conns]
                      :where [:= :conn-id conn-id]
                      :fields [:state :type]}))

(s/defn get-conn :- (s/maybe {s/Keyword s/Any})
  [db :- DBType
   conn-id :- ConnId]
  (fdb/select-one db {:tables [:conns]
                      :where [:= :conn-id conn-id]}))

;; If the conn is closed, the :conns row will not be in the db, so
;; this might return nil
(s/defn get-sender-and-conn-type :- (s/maybe
                                     [(s/one (s/maybe (s/=> s/Any ByteArray))
                                             "sender")
                                      (s/one s/Keyword "conn-type")])
  [db :- DBType
   conn-id :- ConnId]
  (fdb/select-one db {:tables [:conns]
                      :fields [:sender :type]
                      :where [:= :conn-id conn-id]}))

(s/defn get-closer :- (s/maybe (s/=> nil))
  [db :- DBType
   conn-id :- ConnId]
  (fdb/select-one db {:tables [:conns]
                      :fields :closer
                      :where [:= :conn-id conn-id]}))

(s/defn set-sender-closer :- DBType
  [db :- DBType
   conn-id :- ConnId
   sender :- (s/=> s/Any)
   closer :- (s/=> s/Any)]
  (fdb/update db :conns {:set (sym-map sender closer)
                         :where [:= :conn-id conn-id]}))

(s/defn set-conn-state :- DBType
  [db :- DBType
   conn-id :- ConnId
   state :- ConnState]
  (when-not (mu/valid-conn-states state)
    (throw-far-error (str "Invalid connection state `" state "`")
                     :illegal-argument :invalid-connection-state
                     (sym-map state)))
  (fdb/update db :conns {:set (sym-map state)
                         :where [:= :conn-id conn-id]}))

(s/defn get-conn-ids :- [s/Str]
  [db :- DBType]
  (fdb/select db {:tables [:conns]
                  :fields :conn-id}))

(defn- make-fn-schema [service-name fn-name schema-type schema]
  (let [name (join "." [service-name fn-name schema-type schema])]
    (if (map? schema)
      (assoc schema :name name)
      {:name name
       :type schema})))

(s/defn get-schema-rq-time-ms :- (s/maybe Timestamp)
  [db :- DBType
   fingerprint :- Fingerprint]
  (fdb/select-one db {:tables [:schema-rqs]
                      :fields :insert-time-ms
                      :where [:= :fingerprint fingerprint]}))

(s/defn add-schema-rq :- DBType
  [db :- DBType
   fingerprint :- Fingerprint]
  (when-not (get-schema-rq-time-ms db fingerprint)
    (let [insert-time-ms (u/get-current-time-ms)]
      (fdb/insert db :schema-rqs (sym-map fingerprint insert-time-ms)))))

(s/defn delete-schema-rq :- DBType
  [db :- DBType
   fingerprint :- Fingerprint]
  (fdb/delete db :schema-rqs [:= :fingerprint fingerprint]))

(s/defn timed-out? :- s/Bool
  [db :- DBType
   conn-id :- ConnId]
  (let [start-time-ms (fdb/select-one db {:tables [:conns]
                                          :fields :start-time-ms
                                          :where [:= :conn-id conn-id]})]
    (< (+ start-time-ms connect-timeout-ms)
       (u/get-current-time-ms))))

(s/defn get-logged-in-conn-ids :- (s/maybe [ConnId])
  [db :- DBType]
  (fdb/select db {:tables [:conns]
                  :fields :conn-id
                  :where [:= :state :logged-in]
                  :order-by [:conn-id :asc]}))

(s/defn get-gw-conn-id :- (s/maybe ConnId)
  [db :- DBType]
  (first (get-logged-in-conn-ids db)))

(s/defn get-non-logged-in-conn-ids :- (s/maybe [ConnId])
  [db :- DBType]
  (fdb/select db {:tables [:conns]
                  :fields :conn-id
                  :where [:not [:= :state :logged-in]]
                  :order-by [:conn-id :asc]}))

(s/defn add-conn :- DBType
  [db :- DBType
   conn-id :- ConnId
   sender :- (s/maybe (s/=> s/Any))
   closer :- (s/maybe (s/=> s/Any))
   type :- ProcType]
  (let [state :start
        start-time-ms (u/get-current-time-ms)
        row (sym-map conn-id sender closer type state start-time-ms)]
    (fdb/insert db :conns row)))

(s/defn delete-conn :- DBType
  [db :- DBType
   conn-id :- ConnId]
  (fdb/delete db :conns [:= :conn-id conn-id]))

(s/defn delete-fragments :- DBType
  [db :- DBType
   msg-id :- MsgId]
  (fdb/delete db :fragments [:= :msg-id msg-id]))

(s/defn add-schema :- DBType
  [db :- DBType
   fingerprint :- Fingerprint
   json-schema :- s/Str]
  (let [edn-schema (roe/json-schema->edn-schema json-schema)]
    (add-edn-schema db fingerprint edn-schema)))

(s/defn delete-waiting-bytes :- DBType
  [db :- DBType
   fingerprint :- Fingerprint]
  (fdb/delete db :bytes-waiting-for-schemas [:= :fingerprint fingerprint]))

(defn get-threshold-ms [max-wait-ms]
  (- (u/get-current-time-ms) max-wait-ms))

(s/defn gc-fragments :- DBType
  [db :- DBType]
  (let [max-time-ms (get-threshold-ms max-fragment-wait-ms)]
    (fdb/delete db :fragments [:< :insert-time-ms max-time-ms])))

(s/defn gc-schema-rqs :- DBType
  [db :- DBType]
  (let [max-time-ms (get-threshold-ms max-fragment-wait-ms)]
    (fdb/delete db :schema-rqs [:< :insert-time-ms max-time-ms])))

(s/defn gc-msgs-waiting :- DBType
  [db :- DBType]
  (let [max-time-ms (get-threshold-ms max-fragment-wait-ms)]
    (fdb/delete db :bytes-waiting-for-schemas [:< :insert-time-ms
                                               max-time-ms])))

(def op->f
  (sym-map add-conn add-fragment add-schema add-schema-rq gc-fragments
           delete-schema-rq gc-msgs-waiting
           gc-schema-rqs insert-bytes-waiting-for-schemas
           delete-conn delete-fragments delete-waiting-bytes set-conn-state
           set-sender-closer))
