(ns farbetter.roe.prismatic
  (:require
   [farbetter.roe.schemas :as rs :refer
    [AvroFixedOrBytes AvroLong AvroSchema avro-primitive-types]]
   [farbetter.roe.serdes :as serdes]
   [farbetter.roe.utils :as ru]
   [farbetter.utils :as u :refer
    [throw-far-error #?@(:clj [inspect sym-map])]]
   [schema.core :as s :include-macros true]
   [taoensso.timbre :as timbre
    #?(:clj :refer :cljs :refer-macros) [debugf errorf infof tracef]])
  #?(:cljs
     (:require-macros
      [farbetter.utils :refer [inspect sym-map]])))

(declare xf-schema xf-array xf-enum xf-fixed xf-map
         xf-record)

(def LongOrNum (s/if u/long?
                 AvroLong
                 s/Num))

(defn xf-primitive [primitive context]
  (case primitive
    :null (s/eq nil)
    :boolean s/Bool
    :int s/Int
    :long LongOrNum
    :float s/Num
    :double s/Num
    :bytes rs/AvroStringOrBytes
    :string rs/AvroStringOrBytes))

(defn unwrapped-schema->conditionals [schema context]
  (let [{:keys [type]} schema]
    (cond
      (avro-primitive-types schema)
      [(serdes/make-pred schema context) (xf-primitive schema context)]

      (avro-primitive-types type)
      [(serdes/make-pred type context) (xf-primitive type context)]

      (= :map type)
      [serdes/valid-map? (xf-map schema context)]

      (= :record type)
      [serdes/valid-record? (xf-record schema context)]

      (= :enum type)
      [#(some #{%} (:symbols schema)) (xf-enum schema context)]

      (= :array type)
      [sequential? (xf-array schema context)]

      (= :fixed type)
      [#(and (u/byte-array? %)
             (= (:size schema) (count %))) (xf-fixed schema context)]

      :else (if-let [schema (-> context :name->schema schema)]
              (recur schema context)
              (throw-far-error "Unknown schema type."
                               :illegal-schema :unknown-schema-type
                               (sym-map schema type))))))

(defn wrapped-schema->conditionals [schema context]
  (let [pred (fn [data]
               (if-not (and (map? data)
                            (= 1 (count data)))
                 false
                 (let [[data-name un-wrapped-data] (first data)
                       schema-name (ru/get-schema-name schema)]

                   (= schema-name data-name))))
        prismatic-schema {(s/required-key (ru/get-schema-name schema))
                          (xf-schema schema context)}]
    [pred prismatic-schema]))

(defn xf-union [union-schema context]
  (let [wrapped? (ru/union-wrap-reqd? union-schema)
        get-conditionals (if wrapped?
                           wrapped-schema->conditionals
                           unwrapped-schema->conditionals)
        add-schema (fn [acc schema]
                     (concat acc (get-conditionals schema context)))
        arglist (cond-> (vec (reduce add-schema [] union-schema))
                  wrapped? (conj 'wrapped-data?))]
    (apply s/conditional arglist)))

(defn xf-primitive-map [schema context]
  (xf-primitive (:type schema) context))

(defn xf-record  [schema context]
  (let [add-field (fn [acc {:keys [name type]}]
                    (if (and
                         (sequential? type)
                         (= :null (first type)))
                      (let [xf-type (if (= 2 (count type))
                                      (xf-schema (second type) context)
                                      (xf-union (rest type) context))]
                        (assoc acc (s/optional-key name) (s/maybe xf-type)))
                      (let [xf-type (xf-schema type context)]
                        (assoc acc (s/required-key name) xf-type))))]
    (reduce add-field {} (:fields schema))))

(defn xf-enum
  [{:keys [symbols]} context]
  (apply s/enum symbols))

(defn xf-array
  [schema context]
  (let [items-schema (xf-schema (:items schema) context)]
    [items-schema]))

(defn xf-map [schema context]
  (let [vals-schema (xf-schema (:values schema) context)]
    {s/Str vals-schema}))

(defn xf-fixed [schema context]
  rs/AvroStringOrBytes)

(defn xf-schema [schema context]
  (let [xf (loop [schema schema
                  context context]
             (let [{:keys [type]} schema]
               (cond
                 (ru/union? schema) xf-union
                 (avro-primitive-types schema) xf-primitive
                 (avro-primitive-types type) xf-primitive-map
                 (= :record type) xf-record
                 (= :enum type) xf-enum
                 (= :array type) xf-array
                 (= :map type) xf-map
                 (= :fixed type) xf-fixed
                 :else (if-let [schema (-> context :name->schema schema)]
                         (recur schema context)
                         (throw-far-error "Unknown schema type."
                                          :illegal-schema :unknown-schema-type
                                          (sym-map schema type))))))]
    (xf schema context)))

(defn edn-schema->prismatic-schema [schema]
  (let [context (serdes/make-context schema)]
    (xf-schema schema context)))
