(ns farbetter.roe.io-streams
  (:require
   [farbetter.utils :as u :refer [throw-far-error inspect sym-map]])
  (:import
   [com.google.common.io LittleEndianDataInputStream
    LittleEndianDataOutputStream]
   [java.io ByteArrayInputStream ByteArrayOutputStream]
   [java.math BigInteger]
   [java.nio ByteBuffer ByteOrder]
   [java.util Base64]))

(defrecord MutableOutputStream [baos ledos])

(defn write-long-varint-zz [mos l]
  (let [l (if (instance? BigInteger l)
            (.longValue ^BigInteger l)
            l)
        zz-n (bit-xor (bit-shift-left l 1) (bit-shift-right l 63))]
    (loop [n zz-n]
      (if (zero? (bit-and n -128))
        (let [b (bit-and n 0x7f)]
          (.writeByte ^LittleEndianDataOutputStream (:ledos mos) b))
        (let [b (-> (bit-and n 0x7f)
                    (bit-or 0x80))]
          (.writeByte ^LittleEndianDataOutputStream (:ledos mos) b)
          (recur (unsigned-bit-shift-right n 7)))))))

(defn get-size [mos]
  (.size ^ByteArrayOutputStream (:baos mos)))

(defn write-byte [mos b]
  (.writeByte ^LittleEndianDataOutputStream (:ledos mos) b))

(defn write-bytes [mos bs num-bytes]
  (.write ^LittleEndianDataOutputStream (:ledos mos) bs 0 num-bytes))

(defn write-bytes-w-len-prefix [mos bs]
  (let [num-bytes (count bs)]
    (write-long-varint-zz mos num-bytes)
    (write-bytes mos bs num-bytes)))

(defn write-utf8-string [mos s]
  (let [bytes (.getBytes ^String s "UTF-8")
        num-bytes (count bytes)]
    (write-long-varint-zz mos num-bytes)
    (write-bytes mos bytes num-bytes)))

(defn write-float [mos f]
  (.writeFloat ^LittleEndianDataOutputStream (:ledos mos) ^float f))

(defn write-double [mos d]
  (.writeDouble ^LittleEndianDataOutputStream (:ledos mos) ^double d))

(defn to-byte-array [mos]
  (.toByteArray ^ByteArrayOutputStream (:baos mos)))

(defn to-b64-string [mos]
  (let [b64-encoder (Base64/getEncoder)]
    (->> (.toByteArray ^ByteArrayOutputStream (:baos mos))
         (.encodeToString b64-encoder))))

(defn to-utf8-string [mos]
  (String. #^bytes (to-byte-array mos) "UTF-8"))


(defrecord MutableInputStream [bais ledis])

(defn read-long-varint-zz [mis]
  (loop [i 0
         out 0]
    (let [b (.readByte ^LittleEndianDataInputStream (:ledis mis))]
      (if (zero? (bit-and b 0x80))
        (let [zz-n (-> (bit-shift-left b i)
                       (bit-or out))
              long-out (->> (bit-and zz-n 1)
                            (- 0)
                            (bit-xor (unsigned-bit-shift-right zz-n 1)))]
          long-out)
        (let [out (-> (bit-and b 0x7f)
                      (bit-shift-left i)
                      (bit-or out))
              i (+ 7 i)]
          (if (<= i 63)
            (recur i out)
            (throw-far-error "Variable-length quantity is more than 64 bits"
                             :invalid-data :var-len-num-more-than-64-bits
                             (sym-map i))))))))

(defn get-available-count [mis]
  (.available ^ByteArrayInputStream (:bais mis)))

(defn read-byte [mis]
  (.readByte ^LittleEndianDataInputStream (:ledis mis)))

(defn read-bytes [mis num-bytes]
  (let [ba (byte-array num-bytes)]
    (.readFully ^LittleEndianDataInputStream (:ledis mis) #^bytes ba 0 num-bytes)
    ba))

(defn read-len-prefixed-bytes [mis]
  (let [num-bytes (read-long-varint-zz mis)]
    (read-bytes mis num-bytes)))

(defn read-utf8-string [mis]
  (String. #^bytes (read-len-prefixed-bytes mis) "UTF-8"))

(defn read-float [mis]
  (.readFloat ^LittleEndianDataInputStream (:ledis mis)))

(defn read-double [mis]
  (.readDouble ^LittleEndianDataInputStream (:ledis mis)))



(defn byte-array->mutable-input-stream [bs]
  (let [bais (ByteArrayInputStream. bs)
        ledis (LittleEndianDataInputStream. bais)]
    (->MutableInputStream bais ledis)))

(defn b64-string->mutable-input-stream [s]
  (let [b64-decoder (Base64/getDecoder)
        bs (.decode b64-decoder ^String s)
        bais (ByteArrayInputStream. bs)
        ledis (LittleEndianDataInputStream. bais)]
    (->MutableInputStream bais ledis)))

(defn make-mutable-output-stream []
  (let [baos (ByteArrayOutputStream.)
        ledos (LittleEndianDataOutputStream. baos)]
    (->MutableOutputStream baos ledos)))
