(ns farbetter.string-store.impl
  (:require
   [clojure.core.async :as ca]
   [farbetter.string-store.storage :as storage]
   [farbetter.roe :as roe]
   [farbetter.utils :as u :refer [sym-map]]
   [schema.core :as s]
   [taoensso.timbre :as timbre :refer [debugf errorf infof tracef warnf]])
  #?(:clj
     (:import
      [java.nio ByteBuffer])))

(defn get-chunk [storage index chunk-id]
  (u/go-sf
   (let [row (u/call-sf! storage/get-row storage chunk-id)
         {:keys [chunk]} row
         chunk #?(:clj (if (instance? ByteBuffer chunk)
                         (.array chunk)
                         chunk)
                  :cljs chunk)]
     [index chunk])))

(defn get-chunks [storage chunk-ids]
  (u/go-sf
   (let [chans (map-indexed (partial get-chunk storage)
                            chunk-ids)
         result-chan (ca/merge chans)
         result (atom (sorted-map))]
     (dotimes [n (count chunk-ids)]
       (let [[status ret] (ca/<! result-chan)]
         (if (= :success status)
           (let [[index chunk] ret]
             (swap! result assoc index chunk))
           (let [reason ret]
             (if (instance? #?(:clj Throwable :cljs js/Error) reason)
               (throw reason)
               (u/throw-far-error
                "get-chunk failed"
                :execution-error :search-failure
                (sym-map status reason)))))))
     (vals @result))))

(defn load-string* [storage name stored-version]
  (u/go-sf
   (let [stored-version (or stored-version
                            (u/call-sf! storage/get-version storage name))]
     (when stored-version
       (let [header-row-id (storage/make-row-id name stored-version)
             header-row (u/call-sf! storage/get-row storage header-row-id)]
         (when (seq header-row)
           (let [{:keys [chunk-ids]} header-row
                 chunks (u/call-sf! get-chunks storage chunk-ids)
                 deflated-bytes (u/concat-byte-arrays chunks)
                 inflated-bytes (u/inflate deflated-bytes)
                 string (roe/avro-byte-array->edn :string :string
                                                  inflated-bytes)]
             (sym-map string stored-version))))))))

(defn byte-array->chunks [chunk-size ba]
  (loop [offset 0
         output []]
    (if (>= offset (count ba))
      output
      (let [end-offset (+ offset chunk-size)
            chunk (u/slice ba offset end-offset)]
        (recur end-offset
               (conj output chunk))))))

(defn make-uuid-str []
  (-> (u/make-v4-uuid)
      (u/uuid->hex-str)))

(defn store-chunks [storage name stored-version chunks]
  (u/go-sf
   (let [chunk-ids (mapv (fn [_]
                           (make-uuid-str))
                         chunks)
         timestamp (u/get-current-time-ms)
         new-version (u/call-sf! storage/put-header-row storage name
                                 stored-version timestamp chunk-ids)]
     (when new-version
       (doseq [[id chunk] (map vector chunk-ids chunks)]
         (u/call-sf! storage/put-chunk-row storage id (sym-map chunk)))
       new-version))))

(defn store-string* [storage name stored-version s]
  (u/go-sf
   (let [inflated-bytes (roe/edn->avro-byte-array :string s)
         deflated-bytes (u/deflate inflated-bytes)
         chunk-size (u/call-sf! storage/get-chunk-size storage)
         chunks (byte-array->chunks chunk-size deflated-bytes)
         new-version (u/call-sf! store-chunks storage name stored-version
                                 chunks)]
     new-version)))
