(ns farbetter.string-store.mem-storage
  (:require
   [farbetter.string-store.storage :as storage]
   [farbetter.utils :as u :refer [sym-map]]
   [taoensso.timbre :as timbre :refer [debugf errorf infof tracef warnf]]))

(defn get-max-version [existing-storage name]
  (let [row-ids (filter (fn [row-id]
                          (clojure.string/starts-with? row-id name))
                        (keys existing-storage))
        row-nums (map (fn [row-id]
                        (let [[_ id] (clojure.string/split
                                      row-id #"#")]
                          (u/str->int id)))
                      row-ids)]
    (when (seq row-nums)
      (apply max row-nums))))

(defrecord MemStorage [m chunk-size]
  storage/IStorage
  (get-version [this name]
    (u/go-sf
     (get-max-version @m name)))

  (get-row [this name]
    (u/go-sf
     (@m name)))

  (put-header-row [this name stored-version timestamp chunk-ids]
    (u/go-sf
     (let [old-val @m
           max-version (get-max-version old-val name)]
       (when (= stored-version max-version)
         (let [new-version (if (nil? stored-version)
                             0
                             (inc stored-version))
               new-val (assoc old-val
                              (storage/make-row-id name new-version)
                              (sym-map chunk-ids))
               res (compare-and-set! m old-val new-val)]
           ;; Returns nil if failure; new version if successful
           (when res
             new-version))))))

  (put-chunk-row [this row-id chunk]
    (u/go-sf
     (swap! m assoc row-id chunk)))

  (get-chunk-size [this]
    (u/go-sf
     chunk-size)))

(defn make-mem-storage
  ([]
   (make-mem-storage (* 1024 64)))
  ([chunk-size]
   (->MemStorage (atom {}) chunk-size)))
